import { type Face, type FacePosition, type IDSimilarity, type Point, type TrackerID } from './NativeFaceSDK';
import { ERROR, FACIAL_FEATURE_COUNT, FEATURE, FSDKError, type FacialAttribute, type FlatType, IMAGEMODE, ON_ERROR, type Parameter, type ParameterValue, type ParameterValueType, type Parameters, type TrackerFacialAttribute, type TrackerParameter, type TrackerParameters, VIDEOCOMPRESSIONTYPE } from './definitions';
import FSDKWorklets from './FaceSDKWorklets';
export { ERROR, FACIAL_FEATURE_COUNT, FEATURE, FSDKError, IMAGEMODE, ON_ERROR, VIDEOCOMPRESSIONTYPE, type Face, type FacePosition, type FacialAttribute, type IDSimilarity, type Parameter, type ParameterValue, type Parameters, type Point, type TrackerFacialAttribute, type TrackerID, type TrackerParameter, type TrackerParameters };
export interface FaceImage {
    image: Image;
    features: Point[];
}
declare const AgeGroups: readonly ["0-2", "4-6", "8-13", "15-20", "25-32", "38-43", "48-53", "60-"];
declare const Angles: readonly ["Roll", "Pan", "Tilt"];
declare const Expression: readonly ["Smile", "EyesOpen"];
declare const Gender: readonly ["Male", "Female"];
type FacialAttributeKeyLists = {
    AgeGroups: typeof AgeGroups;
    Gender: typeof Gender;
    Expression: typeof Expression;
    Angles: typeof Angles;
};
type FacialAttributeTypes = {
    LivenessError: string;
};
type FacialAttributeKeys<T extends TrackerFacialAttribute> = T extends keyof FacialAttributeKeyLists ? FacialAttributeKeyLists[T] : [T];
type FacialAttributeType<T> = T extends keyof FacialAttributeTypes ? FacialAttributeTypes[T] : number;
type FacialAttributesResultsHelper<T extends TrackerFacialAttribute[]> = {
    [K in keyof T as T[K] extends TrackerFacialAttribute ? FacialAttributeKeys<T[K]>[number] : never]: any;
};
type FacialAttributesResults<T extends TrackerFacialAttribute[]> = {
    [K in keyof FacialAttributesResultsHelper<T>]: FacialAttributeType<K>;
};
/**
 * A byte buffer object to be used with FSDK function. Allows access as a base64 string and ArrayBuffer.
 */
export declare class Buffer {
    private base64;
    private buffer;
    protected constructor(base64?: string | null, buffer?: ArrayBuffer | null);
    /**
     * Create a Buffer object from a base64 encoded buffer.
     * @param {string} base64 the encoded buffer.
     * @returns {Buffer} The created Buffer object.
     */
    static FromBase64(base64: string): Buffer;
    /**
     * Create a Buffer object from an ArrayBuffer object.
     * @param {ArrayBuffer} buffer the buffer.
     * @returns {Buffer} The created Buffer object.
     */
    static FromArrayBuffer(buffer: ArrayBuffer): Buffer;
    /**
     * Obtain the buffer as a base64 string.
     * @returns {string} base64 encoded buffer.
     */
    asBase64(): string;
    /**
     * Obtain the buffer as an ArrayBuffer object.
     * @returns {ArrayBuffer} The buffer.
     */
    asArrayBuffer(): ArrayBuffer;
}
/**
 * A face template representing a person. Allows access as a base64 string and ArrayBuffer.
 */
export declare class FaceTemplate extends Buffer {
    /**
     * Create a FaceTemplate object from a base64 encoded buffer.
     * @param {string} base64 the encoded buffer.
     * @returns {Buffer} The created FaceTemplate object.
     */
    static FromBase64(base64: string): FaceTemplate;
    /**
     * Create a FaceTemplate object from an ArrayBuffer object.
     * @param {ArrayBuffer} buffer the buffer.
     * @returns {Buffer} The created FaceTemplate object.
     */
    static FromBuffer(buffer: ArrayBuffer): FaceTemplate;
    /**
     * Get the similarity score between this template and another.
     * @param {FaceTemplate} other The template to match with.
     * @returns {number} The similarity score.
     */
    match(other: FaceTemplate): number;
}
/**
 * A base64 encoded buffer.
 */
type Base64 = string;
/**
 * An object that can be enterpreted as a byte buffer. Can be either a base64 {@type string}, an {@type ArrayBuffer} or a {@type Buffer} object.
 */
type BufferLike = Buffer | Base64 | ArrayBuffer;
/**
 * A wrapper object for a native handle.
 */
declare class FSDKObject {
    handle: number;
    constructor(handle?: number);
    isValid(): boolean;
}
/**
 * A wrapper object for a FSDK image.
 */
export declare class Image extends FSDKObject {
    /**
     * Create an empty image.
     * @returns {Image} The image.
     */
    static Empty(): Image;
    /**
     * Open an image from a file. PNG, JPG and BMP formats are supported.
     * @param {string} filename The path to the image file.
     * @returns {Image} The image.
     */
    static FromFile(filename: string): Image;
    /**
     * Open an image from a file preserving the alpha channel. PNG, JPG and BMP formats are supported.
     * @param {string} filename The path to the image file.
     * @returns {Image} The image.
     */
    static FromFileWithAlpha(filename: string): Image;
    /**
     * Load an image from a pixel byte buffer. The buffer encodes the image top to bottom with pixel stride equal to pixel byte size and row stride equal to {@param scanLine}.
     * @param {BufferLike} buffer The image buffer.
     * @param {number} width The width of the image.
     * @param {number} height The height of the image.
     * @param {number} scanLine The number of bytes per image row -- row stride.
     * @param {IMAGEMODE} imageMode Image pixel format.
     * @returns {Image} The image.
     */
    static FromBuffer(buffer: BufferLike, width: number, height: number, scanLine: number, imageMode: IMAGEMODE): Image;
    /**
     * Load an image from a JPEG buffer.
     * @param {BufferLike} buffer The image encoded in JPEG format.
     * @returns {Image} The image.
     */
    static FromJpegBuffer(buffer: BufferLike): Image;
    /**
     * Load an image from a PNG buffer.
     * @param {BufferLike} buffer The image encoded in PNG format.
     * @returns {Image} The image.
     */
    static FromPngBuffer(buffer: BufferLike): Image;
    /**
     * Load an image from a PNG buffer preserving the alpha channel.
     * @param {BufferLike} buffer The image encoded in PNG format.
     * @returns {Image} The image.
     */
    static FromPngBufferWithAlpha(buffer: BufferLike): Image;
    /**
     * Save the image into a file specified by {@param filename}.
     * @param {string} filename Path to save the image to.
     * @returns {void}
     */
    saveToFile(filename: string): void;
    /**
     * Get image width.
     * @returns {number} The width of the image.
     */
    getWidth(): number;
    /**
     * Get image height.
     * @returns {number} The height of the image.
     */
    getHeight(): number;
    /**
     * Get the size (in bytes) of a byte buffer that encodes the image in {@param IMAGEMODE} format. Such buffer is guaranteed to have the pixel stride equal to the pixel size in bytes and row stride equal to {@member width} * pixel stride.
     * @param {IMAGEMODE} imageMode The image format to use.
     * @returns {number} The buffer size in bytes.
     */
    getBufferSize(imageMode: IMAGEMODE): number;
    /**
     * Get image width.
     * @returns {number} The width of the image.
     */
    get width(): number;
    /**
     * Get image height.
     * @returns {number} The height of the image.
     */
    get height(): number;
    /**
     * Save image into a byte buffer.
     * @param {IMAGEMODE} imageMode Image format to use.
     * @returns {Buffer} The buffer.
     */
    saveToBuffer(imageMode: IMAGEMODE): Buffer;
    /**
     * Free the internal image buffer. The image becomes invalid.
     * @returns {void}
     */
    free(): void;
    /**
     * Create a copy of the image.
     * @returns {Image} The image copy.
     */
    copy(): Image;
    /**
     * Create a new image by increasing (or decreasing) the total size by {@param ratio}.
     * @param {number} ratio The ratio to change the size of the image by.
     * @returns {Image} The resized image.
     */
    resize(ratio: number): Image;
    /**
     * Create a new image by rotating 90 * {@param multiplier} degrees around the center. Negative values rotate counterclockwise.
     * @param {number} multiplier The number of times to rotate the image 90 degrees.
     * @returns {Image} The rotated image.
     */
    rotate90(multiplier: number): Image;
    /**
     * Create a new image by rotating {@param angle} degrees around the center.
     * @param {number} angle The angle to rotate the image by.
     * @returns {Image} The rotated image.
     */
    rotate(angle: number): Image;
    /**
     * Create a new image by rotating {@param angle} degrees around the point ({@param x}, {@param y}).
     * @param {number} angle The angle to rotate the image by.
     * @param {number} x The x coordinate of the center.
     * @param {number} y The y coordinate of the center.
     * @returns {Image} The rotated image.
     */
    rotateCenter(angle: number, x: number, y: number): Image;
    /**
     * Create a new image by copying an axis aligned rectangle bounded by the top left ({@param x1}, {@param y1}) and the bottom right ({@param x2}, {@param y2}) points.
     * @param {number} x1 The top left point x coordinate.
     * @param {number} y1 The top left point y coordinate.
     * @param {number} x2 The bottom right point x coordinate.
     * @param {number} y2 The bottom right point y coordinate.
     * @returns {Image} The copied image.
     */
    copyRect(x1: number, y1: number, x2: number, y2: number): Image;
    /**
     * Create a new image by copying an axis aligned rectangle bounded by the top left ({@param x1}, {@param y1}) and the bottom right ({@param x2}, {@param y2}) points. The part of the copy that lies outside of the image will be repeated from the border.
     * @param {number} x1 The top left point x coordinate.
     * @param {number} y1 The top left point y coordinate.
     * @param {number} x2 The bottom right point x coordinate.
     * @param {number} y2 The bottom right point y coordinate.
     * @returns {Image} The copied image.
     */
    copyRectReplicateBorder(x1: number, y1: number, x2: number, y2: number): Image;
    /**
     * Mirror the image.
     * @param {boolean} vertical Whether mirroring is performed around the vertical or horizontal axis.
     * @returns {void}
     */
    mirror(vertical?: boolean): void;
    /**
     * Extract the part of the image that contains the face specified by {@param features} and resize it to {@param width}x{@param height}.
     * @param {Point[]} features Array of face key points.
     * @param {number} width Target image width.
     * @param {height} height Target image height.
     * @returns {FaceImage} The extracted part of the image and new features.
     */
    extractFace(features: Point[], width: number, height: number): FaceImage;
    /**
     * Detect a single face in the image. If multiple faces are present returns the one with the highest detection score.
     * @returns {FacePosition} The detected face.
     */
    detectFace(): FacePosition;
    /**
     * Detect multiple faces in the image. The faces are sorted by detection score in descending order.
     * @param {number} maxFaces The maximal number of faces to detect.
     * @returns {FacePosition[]} The detected faces.
     */
    detectMultipleFaces(maxFaces?: number): FacePosition[];
    /**
     * Detect 70 facial key points of a single face in the image. If multiple faces are present detects points for the face with the highest detection score.
     * @returns {Point[]} The detected key points.
     */
    detectFacialFeatures(): Point[];
    /**
     * Detect 70 facial key points for a given face {@param position}.
     * @param {FacePosition} position The face to detect key points for.
     * @returns {Point[]} The detected key points.
     */
    detectFacialFeaturesInRegion(position: FacePosition): Point[];
    /**
     * Detect the corrdinates of eye centers of a single face in the image. If multiple faces are present detects points for the face with the highest detection score.
     * @returns {Point[]} The detected eyes points.
     */
    detectEyes(): Point[];
    /**
     * Detect coordinates of eyes for a given face {@param position}.
     * @param {FacePosition} position The face to detect eyes for.
     * @returns {Point[]} The detected eyes points.
     */
    detectEyesInRegion(position: FacePosition): Point[];
    /**
     * Get face template of a single face in the image. If multiple faces are present the face with the highest detection score is considered.
     * @returns {FaceTemplate} The face template.
     */
    getFaceTemplate(): FaceTemplate;
    /**
     * Get face template for a given face {@param position}.
     * @param {FacePosition} position The face to get template for.
     * @returns {FaceTemplate} The face template.
     */
    getFaceTemplateInRegion(position: FacePosition): FaceTemplate;
    /**
     * Get face template given facial keypoints.
     * @param {Point[]} features The face to get template for.
     * @returns {FaceTemplate} The face template.
     */
    getFaceTemplateUsingFeatures(features: Point[]): FaceTemplate;
    /**
     * Get face template given eye coodinates.
     * @param {Point[]} eyes Coordinates of the eye centers.
     * @returns {FaceTemplate} The face template.
     */
    getFaceTemplateUsingEyes(eyes: Point[]): FaceTemplate;
    /**
     * Detect facial attribute values (i.e. angles, liveness) using facial keypoints.
     * @template {FacialAttribute[]} A
     * @param {Point[]} features Array of facial keypoints.
     * @param {A} attributes The attributes to detect.
     * @returns {Object} An object with attribute values. The obejct keys depend on which attriutes are requested.
     */
    detectFacialAttributeUsingFeatures<A extends FacialAttribute[]>(features: Point[], ...attributes: A): FlatType<FacialAttributesResults<A>>;
    /**
     * Detect facial attribute values (i.e. angles, liveness) using facial keypoints.
     * @param {Point[]} features Array of facial keypoints.
     * @param {string | string[]} attribute The attribute (or array of attributes) to detect.
     * @param {string} maxSize The maximal size of the returned string.
     * @returns {string} A string with the detected attribute values in key1=value1;key2=value2; format.
     */
    detectFacialAttributeUsingFeaturesRaw(features: Point[], attribute: string | string[], maxSize?: number): string;
}
/** A wrapper object for a FSDK Tracker */
export declare class Tracker extends FSDKObject {
    /**
     * Create an empty tracker.
     * @returns {Tracker} The tracker.
     */
    static Create(): Tracker;
    /**
     * Create a tracker and load its memory from file.
     * @param {string} filename Path to the tracker memory file.
     * @returns {Tracker} The tracker.
     */
    static FromFile(filename: string): Tracker;
    /**
     * Create a tracker and load its memory from {@param buffer}.
     * @param {BufferLike} buffer A buffer with tracker memory.
     * @returns {Tracker} The tracker.
     */
    static FromBuffer(buffer: BufferLike): Tracker;
    /**
     * Free the tracker. The tracker becomes invalid.
     * @returns {void}
     */
    free(): void;
    /**
     * Clear tracker memory. Sets all parameters to their default values and clears any stored faces.
     * @returns {void}
     */
    clear(): void;
    /**
     * Save tracker memory to file. Including parameters and faces.
     * @param {string} filename Path to save memory to.
     * @returns {void}
     */
    saveToFile(filename: string): void;
    /**
     * Get the size (in bytes) of a byte buffer that is enough to store the tracker memory.
     * @returns {number} The buffer size in bytes.
     */
    getMemoryBufferSize(): number;
    /**
     * Save tracker memory to buffer.
     * @returns {Buffer} The buffer.
     */
    saveToBuffer(): Buffer;
    /**
     * Set tracker parameter.
     * @template {TrackerParameter} P
     * @param {P} name Tracker parameter name.
     * @param {TrackerParameterValueType<P>} value Tracker parameter value.
     * @returns {void}
     */
    setParameter<P extends TrackerParameter>(name: P, value: ParameterValueType<P>): void;
    /**
     * Set tracker parameter.
     * @param {TrackerParameter} name Tracker parameter name.
     * @param {string} value Tracker parameter value.
     * @returns {void}
     */
    setParameterRaw(name: string, value: string): void;
    /**
     * Set multiple tracker parameters.
     * @template {TrackerParameter} P Tracker parameters to set.
     * @param {string | TrackerParameters<P>} parameters Parameters to set. If string, must have the following format 'key1=value1;key2=value2;...'
     * @returns {number} Index of the parsing error position if any.
     */
    setMultipleParameters<P extends TrackerParameter>(parameters: string | TrackerParameters<P>): number;
    /**
     * Get tracker parameter value.
     * @template {TrackerParameter} P
     * @param {P} parameter The parameter to get the value of.
     * @param {number} maxSize The max size of the internal string buffer for parameter value retrieval.
     * @returns {FlatType<ParameterValueType<P>>} The parameter value.
     */
    getParameter<P extends TrackerParameter>(parameter: P, maxSize?: number): ParameterValueType<P>;
    /**
     * Process an image using tracker, which includes face detection, recognition, tracking, facial features and attributes detection.
     * @param {Image} image The image to process.
     * @param {number} maxFaces Maximal number of faces to process.
     * @param {number} index Camera index (unused).
     * @returns {number[]} Array of detected ids.
     */
    feedFrame(image: Image, maxFaces?: number, index?: number): number[];
    /**
     * Get eye coordinates for a id.
     * @param {number} id Id of the face to get eye coordinates for.
     * @param {number} index Camera index (unused).
     * @returns {Point[]} Array of two points -- the eye coordinates.
     */
    getEyes(id: number, index?: number): Point[];
    /**
     * Get facial feature coordinates for a id.
     * @param {number} id Id of the face to get facial feature coordinates for.
     * @param {number} index Camera index (unused).
     * @returns {Point[]} Array of 70 points -- the facial feature coordinates.
     */
    getFacialFeatures(id: number, index?: number): Point[];
    /**
     * Get face position for a id.
     * @param {number} id Id of the face to get face position for.
     * @param {number} index Camera index (unused).
     * @returns {FacePosition} The face position.
     */
    getFacePosition(id: number, index?: number): FacePosition;
    /**
     * Lock a face id. Required for any operation with the id (i.e. getting or setting the name).
     * @param {number} id Face id to lock.
     * @returns {void}
     */
    lockID(id: number): void;
    /**
     * Unlock a id.
     * @param {number} id Face id to unlock.
     * @returns {void}
     */
    unlockID(id: number): void;
    /**
     * Purge a id, deleting any information stored for that id.
     * @param {number} id Face id to purge.
     * @returns {void}
     */
    purgeID(id: number): void;
    /**
     * Set name for a id.
     * @param {number} id Id to set name for.
     * @param {string} name The new name.
     * @returns {void}
     */
    setName(id: number, name: string): void;
    /**
     * Get the name for a id.
     * @param {number} id Id to get the name for.
     * @param {number} maxSize The maximal size of the returned string.
     * @returns {string} The name.
     */
    getName(id: number, maxSize?: number): string;
    /**
     * Get a list of names associated with the id. This includes the name of the id and also the names of similar ids.
     * @param {number} id Id to get the name for.
     * @param {number} maxSize The maximal size of the returned string.
     * @returns {string[]} The list of names.
     */
    getAllNames(id: number, maxSize?: number): string[];
    /**
     * If there was an id merge, return the id that this id was merged into. Otherwise return the same id.
     * @param {number} id Id to get the merged id for.
     * @returns {number} The merged id.
     */
    getIDReassignment(id: number): number;
    /**
     * Get the number of ids similar to this id.
     * @param {number} id Id to get the number of similar ids for.
     * @returns {number} The number of similar ids.
     */
    getSimilarIDCount(id: number): number;
    /**
     * Get the similar ids to this id.
     * @param {number} id Id to get similar ids for.
     * @returns {number[]} Array of ids.
     */
    getSimilarIDList(id: number): number[];
    /**
     * Get the total number of ids stored in tracker memory.
     * @returns {number} The number of ids.
     */
    getIDsCount(): number;
    /**
     * Get all tracker ids.
     * @returns {number[]} Array of tracker ids.
     */
    getAllIDs(): number[];
    /**
     * Get the number of face ids for id.
     * @param {number} id Id to get the number of face ids for.
     * @returns {number} The number of face ids.
     */
    getFaceIDsCountForID(id: number): number;
    /**
     * Get face ids for id.
     * @param {number} id Id to get face ids for.
     * @returns {number[]} Array of face ids.
     */
    getFaceIDsForID(id: number): number[];
    /**
     * Get id for a face id.
     * @param {number} faceID Face id to get id for.
     * @returns {number} The id.
     */
    getIDByFaceID(faceID: number): number;
    /**
     * Get the face template for face id.
     * @param {number} faceID Face id to get face template for.
     * @returns {FaceTemplate} The face template.
     */
    getFaceTemplate(faceID: number): FaceTemplate;
    /**
     * Get the face image for face id.
     * @param {number} faceID Face id to get face image for.
     * @returns {Image} The face image.
     */
    getFaceImage(faceID: number): Image;
    /**
     * Set the face image for face id.
     * @param {number} faceID Face id to set face image for.
     * @param {Image} image The face image.
     * @returns {void}
     */
    setFaceImage(faceID: number, image: Image): void;
    /**
     * Delete the face image for face id.
     * @param {number} faceID Face id to delete the face image for.
     * @returns {void}
     */
    deleteFaceImage(faceID: number): void;
    /**
     * Create a new id in the tracker memory.
     * @param {FaceTemplate} template Face template of the id.
     * @returns {number} The newly created id.
     */
    createID(template: FaceTemplate): TrackerID;
    /**
     * Add a new face template for id.
     * @param {number} id Id to add a new template for.
     * @param {FaceTemplate} template The template.
     * @returns {void}
     */
    addFaceTemplate(id: number, template: FaceTemplate): void;
    /**
     * Delete face id.
     * @param {number} faceID Face id to delete.
     * @returns {void}
     */
    deleteFace(faceID: number): void;
    /**
     * Get ids and their similarities for a face template.
     * @param {FaceTemplate} template The template to find similar ids for.
     * @param {number} threshold Matching similarity threshold for the returned ids.
     * @param {number} maxSize Maximal number of ids to return.
     * @returns {IDSimilarity[]} Array of ids and their similarities.
     */
    matchFaces(template: FaceTemplate, threshold: number, maxSize?: number): IDSimilarity[];
    /**
     * Get facial attribute values (i.e. angles, liveness) for an id.
     * @template {TrackerFacialAttribute[]} A
     * @param {number} id Id to get attributes for.
     * @param {A} attributes The attributes to get.
     * @returns {Object} An object with attribute values. The obejct keys depend on which attriutes are requested.
     */
    getFacialAttribute<A extends TrackerFacialAttribute[]>(id: number, ...attributes: A): FlatType<FacialAttributesResults<A>>;
    /**
     * Get facial attribute values (i.e. angles, liveness) for an id.
     * @param {number} id Id to get attributes for.
     * @param {string | string[]} attribute The attribute (or array of attributes) to get.
     * @param {string} maxSize The maximal size of the returned string.
     * @param {number} index Camera index (unused).
     * @returns {string} A string with the attribute values in key1=value1;key2=value2; format.
     */
    getFacialAttributeRaw(id: number, attribute: string | string[], maxSize?: number, index?: number): string;
}
/** A wrapper object for an IP camera */
export declare class Camera extends FSDKObject {
    /**
     * Set an HTTP proxy to be used with an IP camera.
     * @param {string} address Proxy address.
     * @param {number} port Proxy port.
     * @param {string} username Proxy username.
     * @param {string} password Proxy password.
     * @returns {void}
     */
    static SetHTTPProxy(address: string, port: number, username: string, password: string): void;
    /**
     * Open IP video camera at a given URL.
     * @param {FlatType<VIDEOCOMPRESSIONTYPE>} comression Video compression to use.
     * @param {string} url Camera URL.
     * @param {string} username Camera access username.
     * @param {string} password Camera access password.
     * @param {number} timeout Connection timeout in seconds.
     * @returns {Camera} The IP video camera.
     */
    static OpenIPVideoCamera(comression: FlatType<VIDEOCOMPRESSIONTYPE>, url: string, username: string, password: string, timeout: number): Camera;
    /**
     * Initialize the capturing process.
     * @returns {void}
     */
    static InitializeCapturing(): void;
    /**
     * Finalize the capturing process.
     * @returns {void}
     */
    static FinalizeCapturing(): void;
    /**
     * Obtain the current camera frame image.
     * @returns {Image} The frame image.
     */
    grabFrame(): Image;
    /**
     * Close the camera. Camera becomes invalid.
     * @returns {void}
     */
    close(): void;
}
/** Main FSDK class, exposing all the functions at once */
export default class FSDK {
    static readonly LuxandFaceSDK: import("./NativeFaceSDK").Spec;
    static readonly Image: typeof Image;
    static readonly Buffer: typeof Buffer;
    static readonly Camera: typeof Camera;
    static readonly Tracker: typeof Tracker;
    static readonly FaceTemplate: typeof FaceTemplate;
    static readonly ERROR: typeof ERROR;
    static readonly FEATURE: typeof FEATURE;
    static readonly IMAGEMODE: typeof IMAGEMODE;
    static readonly FACIAL_FEATURE_COUNT: number;
    static readonly VIDEOCOMPRESSIONTYPE: typeof VIDEOCOMPRESSIONTYPE;
    static readonly ON_ERROR: typeof ON_ERROR;
    static onError: ON_ERROR;
    static readonly Worklets: typeof FSDKWorklets;
    /**
     * Set error behaviour.
     * @param {ON_ERROR} value The new on error behaviour.
     * @returns {void}
     */
    static SetOnError(value: ON_ERROR): void;
    /**
     * Activate the library. Must be called before any other FSDK function.
     * @param {string} licenseKey The license key string.
     * @returns {void}
     */
    static ActivateLibrary(licenseKey: string): void;
    /**
     * Initializes the library. Must be called after {@member ActivateLibrary}.
     * @returns {void}
     */
    static Initialize(): void;
    /**
     * Activates and initializes the library. Must be called before any other FSDK function.
     * @param {string} licenseKey The license key string.
     * @returns {void}
     */
    static ActivateAndInitalize(licenseKey: string): void;
    /**
     * Get license information string.
     * @returns {string} The information string.
     */
    static GetLicenseInfo(): string;
    /**
     * Get version information string.
     * @returns {string} The information string.
     */
    static GetVersionInfo(): string;
    /**
     * Create an empty image.
     * @returns {Image} The image.
     */
    static CreateEmptyImage(): Image;
    /**
     * Free the internal image buffer. The image becomes invalid.
     * @param {Image} image The image to free.
     * @returns {void}
     */
    static FreeImage(image: Image): void;
    /**
     * Open an image from a file. PNG, JPG and BMP formats are supported.
     * @param {string} filename The path to the image file.
     * @returns {Image} The image.
     */
    static LoadImageFromFile(filename: string): Image;
    /**
     * Open an image from a file preserving the alpha channel. PNG, JPG and BMP formats are supported.
     * @param {string} filename The path to the image file.
     * @returns {Image} The image.
     */
    static LoadImageFromFileWithAlpha(filename: string): Image;
    /**
     * Load an image from a pixel byte buffer. The buffer encodes the image top to bottom with pixel stride equal to pixel byte size and row stride equal to {@param scanLine}.
     * @param {BufferLike} buffer The image buffer.
     * @param {number} width The width of the image.
     * @param {number} height The height of the image.
     * @param {number} scanLine The number of bytes per image row -- row stride.
     * @param {IMAGEMODE} imageMode Image pixel format.
     * @returns {Image} The image.
     */
    static LoadImageFromBuffer(buffer: BufferLike, width: number, height: number, scanLine: number, imageMode: IMAGEMODE): Image;
    /**
     * Load an image from a JPEG buffer.
     * @param {BufferLike} buffer The image encoded in JPEG format.
     * @returns {Image} The image.
     */
    static LoadImageFromJpegBuffer(buffer: BufferLike): Image;
    /**
     * Load an image from a PNG buffer.
     * @param {BufferLike} buffer The image encoded in PNG format.
     * @returns {Image} The image.
     */
    static LoadImageFromPngBuffer(buffer: BufferLike): Image;
    /**
     * Load an image from a PNG buffer preserving the alpha channel.
     * @param {BufferLike} buffer The image encoded in PNG format.
     * @returns {Image} The image.
     */
    static LoadImageFromPngBufferWithAlpha(buffer: BufferLike): Image;
    /**
     * Save the image into a file specified by {@param filename}.
     * @param {Image} image The image to save.
     * @param {string} filename Path to save the image to.
     * @returns {void}
     */
    static SaveImageToFile(image: Image, filename: string): void;
    /**
     * Set JPEG compression quality used for saving images in JPEG format.
     * @param {number} quality The quality to use.
     * @returns {void}
     */
    static SetJpegCompressionQuality(quality: number): void;
    /**
     * Get image width.
     * @param {Image} image The image to get the width of.
     * @returns {number} The width of the image.
     */
    static GetImageWidth(image: Image): number;
    /**
     * Get image height.
     * @param {Image} image The image to get the height of.
     * @returns {number} The height of the image.
     */
    static GetImageHeight(image: Image): number;
    /**
     * Get the size (in bytes) of a byte buffer that encodes the image in {@param IMAGEMODE} format. Such buffer is guaranteed to have the pixel stride equal to the pixel size in bytes and row stride equal to {@member width} * pixel stride.
     * @param {Image} image The image to get the buffer size of.
     * @param {IMAGEMODE} imageMode The image format to use.
     * @returns {number} The buffer size in bytes.
     */
    static GetImageBufferSize(image: Image, imageMode: IMAGEMODE): number;
    /**
     * Save image into a byte buffer.
     * @param {Image} image Image to save.
     * @param {IMAGEMODE} imageMode Image format to use.
     * @returns {Buffer} The buffer.
     */
    static SaveImageToBuffer(image: Image, imageMode: IMAGEMODE): Buffer;
    /**
     * Create a copy of the image.
     * @param {Image} image The image to create a copy of.
     * @returns {Image} The image copy.
     */
    static CopyImage(image: Image): Image;
    /**
     * Create a new image by increasing (or decreasing) the total size by {@param ratio}.
     * @param {Image} image The image to resize.
     * @param {number} ratio The ratio to change the size of the image by.
     * @returns {Image} The resized image.
     */
    static ResizeImage(image: Image, ratio: number): Image;
    /**
     * Create a new image by rotating 90 * {@param multiplier} degrees around the center. Negative values rotate counterclockwise.
     * @param {Image} image The image to rotate.
     * @param {number} multiplier The number of times to rotate the image 90 degrees.
     * @returns {Image} The rotated image.
     */
    static RotateImage90(image: Image, multiplier: number): Image;
    /**
     * Create a new image by rotating {@param angle} degrees around the center.
     * @param {Image} image The image to rotate.
     * @param {number} angle The angle to rotate the image by.
     * @returns {Image} The rotated image.
     */
    static RotateImage(image: Image, angle: number): Image;
    /**
     * Create a new image by rotating {@param angle} degrees around the point ({@param x}, {@param y}).
     * @param {Image} image The image to rotate.
     * @param {number} angle The angle to rotate the image by.
     * @param {number} x The x coordinate of the center.
     * @param {number} y The y coordinate of the center.
     * @returns {Image} The rotated image.
     */
    static RotateImageCenter(image: Image, angle: number, x: number, y: number): Image;
    /**
     * Create a new image by copying an axis aligned rectangle bounded by the top left ({@param x1}, {@param y1}) and the bottom right ({@param x2}, {@param y2}) points.
     * @param {Image} image The image to copy a rect from.
     * @param {number} x1 The top left point x coordinate.
     * @param {number} y1 The top left point y coordinate.
     * @param {number} x2 The bottom right point x coordinate.
     * @param {number} y2 The bottom right point y coordinate.
     * @returns {Image} The copied image.
     */
    static CopyRect(image: Image, x1: number, y1: number, x2: number, y2: number): Image;
    /**
     * Create a new image by copying an axis aligned rectangle bounded by the top left ({@param x1}, {@param y1}) and the bottom right ({@param x2}, {@param y2}) points. The part of the copy that lies outside of the image will be repeated from the border.
     * @param {Image} image The image to copy a rect from.
     * @param {number} x1 The top left point x coordinate.
     * @param {number} y1 The top left point y coordinate.
     * @param {number} x2 The bottom right point x coordinate.
     * @param {number} y2 The bottom right point y coordinate.
     * @returns {Image} The copied image.
     */
    static CopyRectReplicateBorder(image: Image, x1: number, y1: number, x2: number, y2: number): Image;
    /**
     * Mirror the image.
     * @param {Image} image The image to mirror.
     * @param {boolean} vertical Whether mirroring is performed around the vertical or horizontal axis.
     * @returns {void}
     */
    static MirrorImage(image: Image, vertical?: boolean): void;
    /**
     * Extract the part of the image that contains the face specified by {@param features} and resize it to {@param width}x{@param height}.
     * @param {Image} image The imate to extract a face image from.
     * @param {Point[]} features Array of face key points.
     * @param {number} width Target image width.
     * @param {height} height Target image height.
     * @returns {FaceImage} The extracted part of the image and new features.
     */
    static ExtractFaceImage(image: Image, features: Point[], width: number, height: number): FaceImage;
    /**
     * Detect a single face in the image. If multiple faces are present returns the one with the highest detection score.
     * @param {Image} image The image to detect face on.
     * @returns {FacePosition} The detected face.
     */
    static DetectFace(image: Image): FacePosition;
    /**
     * Detect multiple faces in the image. The faces are sorted by detection score in descending order.
     * @param {Image} image The image to detect faces on.
     * @param {number} maxFaces The maximal number of faces to detect.
     * @returns {FacePosition[]} The detected faces.
     */
    static DetectMultipleFaces(image: Image, maxFaces?: number): FacePosition[];
    /**
     * Set face detection parameters. These do not apply to the improved face detection algorithm.
     * @param {boolean} handleArbitraryRotations Detect rotated faces.
     * @param {boolean} determineFaceRotationAngle Determine the face box rotation angle.
     * @param {number} internalResizeWidth
     * @returns {void}
     */
    static SetFaceDetectionParameters(handleArbitraryRotations: boolean, determineFaceRotationAngle: boolean, internalResizeWidth: number): void;
    /**
     * Set face detection threshold. This does not apply to the improved face detection algorithm.
     * @param {number} threshold The face detection threshold.
     * @returns {void}
     */
    static SetFaceDetectionThreshold(threshold: number): void;
    /**
     * Get the last detected face confidence.
     * @returns {number} The confidence.
     */
    static GetDetectedFaceConfidence(): number;
    /**
     * Detect 70 facial key points of a single face in the image. If multiple faces are present detects points for the face with the highest detection score.
     * @param {Image} image The image to detect features on.
     * @returns {Point[]} The detected key points.
     */
    static DetectFacialFeatures(image: Image): Point[];
    /**
     * Detect 70 facial key points for a given face {@param position}.
     * @param {Image} image The image to detect features on.
     * @param {FacePosition} position The face to detect key points for.
     * @returns {Point[]} The detected key points.
     */
    static DetectFacialFeaturesInRegion(image: Image, position: FacePosition): Point[];
    /**
     * Detect the corrdinates of eye centers of a single face in the image. If multiple faces are present detects points for the face with the highest detection score.
     * @param {Image} image The image to detect eyes on.
     * @returns {Point[]} The detected eyes points.
     */
    static DetectEyes(image: Image): Point[];
    /**
     * Detect coordinates of eyes for a given face {@param position}.
     * @param {Image} image The image to detect eyes on.
     * @param {FacePosition} position The face to detect eyes for.
     * @returns {Point[]} The detected eyes points.
     */
    static DetectEyesInRegion(image: Image, position: FacePosition): Point[];
    /**
     * Get face template of a single face in the image. If multiple faces are present the face with the highest detection score is considered.
     * @param {Image} image The image to get face template for.
     * @returns {FaceTemplate} The face template.
     */
    static GetFaceTemplate(image: Image): FaceTemplate;
    /**
     * Get face template for a given face {@param position}.
     * @param {Image} image The image to get face template for.
     * @param {FacePosition} position The face to get template for.
     * @returns {FaceTemplate} The face template.
     */
    static GetFaceTemplateInRegion(image: Image, position: FacePosition): FaceTemplate;
    /**
     * Get face template given facial keypoints.
     * @param {Image} image The image to get face template for.
     * @param {Point[]} features The face to get template for.
     * @returns {FaceTemplate} The face template.
     */
    static GetFaceTemplateUsingFeatures(image: Image, features: Point[]): FaceTemplate;
    /**
     * Get face template given eye coodinates.
     * @param {Image} image The image to get face template for.
     * @param {Point[]} eyes Coordinates of the eye centers.
     * @returns {FaceTemplate} The face template.
     */
    static GetFaceTemplateUsingEyes(image: Image, eyes: Point[]): FaceTemplate;
    /**
     * Get the similarity score between two face templates.
     * @param {FaceTemplate} template1 First template.
     * @param {FaceTemplate} template1 Second template.
     * @returns {number} The similarity score.
     */
    static MatchFaces(template1: FaceTemplate, template2: FaceTemplate): number;
    /**
     * Get a matching threshold that achieves the gives false acceptance rate ({@param far}).
     * @param {number} far Requested false acceptance rate.
     * @returns {number} The matching threshold.
     */
    static GetMatchingThresholdAtFAR(far: number): number;
    /**
     * Get a matching threshold that achieves the gives false rejection rate ({@param frr}).
     * @param {number} frr Requested false rejection rate.
     * @returns {number} The matching threshold.
     */
    static GetMatchingThresholdAtFRR(frr: number): number;
    /**
     * Create an empty tracker.
     * @returns {Tracker} The tracker.
     */
    static CreatTracker(): Tracker;
    /**
     * Create a tracker and load its memory from file.
     * @param {string} filename Path to the tracker memory file.
     * @returns {Tracker} The tracker.
     */
    static LoadTrackerMemoryFromFile(filename: string): Tracker;
    /**
     * Create a tracker and load its memory from {@param buffer}.
     * @param {BufferLike} buffer A buffer with tracker memory.
     * @returns {Tracker} The tracker.
     */
    static LoadTrackerMemoryFromBuffer(buffer: BufferLike): Tracker;
    /**
     * Free the tracker. The tracker becomes invalid.
     * @param {Tracker} tracker The tracker to free.
     * @returns {void}
     */
    static FreeTracker(tracker: Tracker): void;
    /**
     * Clear tracker memory. Sets all parameters to their default values and clears any stored faces.
     * @param {Tracker} tracker The tracker to clear.
     * @returns {void}
     */
    static ClearTracker(tracker: Tracker): void;
    /**
     * Save tracker memory to file. Including parameters and faces.
     * @param {Tracker} tracker The tracker to save.
     * @param {string} filename Path to save memory to.
     * @returns {void}
     */
    static SaveTrackerMemoryToFile(tracker: Tracker, filename: string): void;
    /**
     * Get the size (in bytes) of a byte buffer that is enough to store the tracker memory.
     * @param {Tracker} tracker The tracker to get buffer size for.
     * @returns {number} The buffer size in bytes.
     */
    static GetTrackerMemoryBufferSize(tracker: Tracker): number;
    /**
     * Save tracker memory to buffer.
     * @param {Tracker} tracker The tracker to save.
     * @returns {Buffer} The buffer.
     */
    static SaveTrackerMemoryToBuffer(tracker: Tracker): Buffer;
    /**
     * Set tracker parameter.
     * @template {TrackerParameter} P
     * @param {Tracker} tracker The tracker to set the parameter for.
     * @param {P} name Tracker parameter name.
     * @param {TrackerParameterValueType<P>} value Tracker parameter value.
     * @returns {void}
     */
    static SetTrackerParameter<P extends TrackerParameter>(tracker: Tracker, name: P, value: ParameterValueType<P>): void;
    /**
     * Set tracker parameter.
     * @param {Tracker} tracker The tracker to set the parameter for.
     * @param {string} name Tracker parameter name.
     * @param {string} value Tracker parameter value.
     * @returns {void}
     */
    static SetTrackerParameterRaw(tracker: Tracker, name: string, value: string): void;
    /**
     * Set multiple tracker parameters.
     * @template {TrackerParameter} T Tracker parameters to set.
     * @param {Tracker} tracker The tracker to set the parameters for.
     * @param {string | TrackerParameters<P>} parameters Parameters to set. If string, must have the following format 'key1=value1;key2=value2;...'
     * @returns {number} Index of the parsing error position if any.
     */
    static SetTrackerMultipleParameters<T extends TrackerParameter>(tracker: Tracker, parameters: string | TrackerParameters<T>): number;
    /**
     * Get tracker parameter value.
     * @template {TrackerParameter} P
     * @param {Tracker} tracker The tracker to set the parameters for.
     * @param {P} parameter The parameter to get the value of.
     * @param {number} maxSize The max size of the internal string buffer for parameter value retrieval.
     * @returns {FlatType<ParameterValueType<P>>} The parameter value.
     */
    static GetTrackerParameter<P extends TrackerParameter>(tracker: Tracker, parameter: P, maxSize?: number): ParameterValueType<P>;
    /**
     * Process an image using tracker, which includes face detection, recognition, tracking, facial features and attributes detection.
     * @param {Tracker} tracker The tracker to feed to frame to.
     * @param {Image} image The image to process.
     * @param {number} maxFaces Maximal number of faces to process.
     * @param {number} index Camera index (unused).
     * @returns {number[]} Array of detected ids.
     */
    static FeedFrame(tracker: Tracker, image: Image, maxFaces?: number, index?: number): number[];
    /**
     * Get eye coordinates for a id.
     * @param {Tracker} tracker The tracker to get eyes from.
     * @param {number} id Id of the face to get eye coordinates for.
     * @param {number} index Camera index (unused).
     * @returns {Point[]} Array of two points -- the eye coordinates.
     */
    static GetTrackerEyes(tracker: Tracker, id: number, index?: number): Point[];
    /**
     * Get facial feature coordinates for a id.
     * @param {Tracker} tracker The tracker to get facial features from.
     * @param {number} id Id of the face to get facial feature coordinates for.
     * @param {number} index Camera index (unused).
     * @returns {Point[]} Array of 70 points -- the facial feature coordinates.
     */
    static GetTrackerFacialFeatures(tracker: Tracker, id: number, index?: number): Point[];
    /**
     * Get face position for a id.
     * @param {Tracker} tracker The tracker to get face position from.
     * @param {number} id Id of the face to get face position for.
     * @param {number} index Camera index (unused).
     * @returns {FacePosition} The face position.
     */
    static GetTrackerFacePosition(tracker: Tracker, id: number, index?: number): FacePosition;
    /**
     * Lock a face id. Required for any operation with the id (i.e. getting or setting the name).
     * @param {Tracker} tracker The tracker to lock the id in.
     * @param {number} id Face id to lock.
     * @returns {void}
     */
    static LockID(tracker: Tracker, id: number): void;
    /**
     * Unlock a id.
     * @param {Tracker} tracker The tracker to unlock the id in.
     * @param {number} id Face id to unlock.
     * @returns {void}
     */
    static UnlockID(tracker: Tracker, id: number): void;
    /**
     * Purge a id, deleting any information stored for that id.
     * @param {Tracker} tracker The tracker to purge the id from.
     * @param {number} id Face id to purge.
     * @returns {void}
     */
    static PurgeID(tracker: Tracker, id: number): void;
    /**
     * Set name for a id.
     * @param {Tracker} tracker The tracker to set the name in.
     * @param {number} id Id to set name for.
     * @param {string} name The new name.
     * @returns {void}
     */
    static SetName(tracker: Tracker, id: number, name: string): void;
    /**
     * Get the name for a id.
     * @param {Tracker} tracker The tracker to get the name from.
     * @param {number} id Id to get the name for.
     * @param {number} maxSize The maximal size of the returned string.
     * @returns {string} The name.
     */
    static GetName(tracker: Tracker, id: number, maxSize?: number): string;
    /**
     * Get a list of names associated with the id. This includes the name of the id and also the names of similar ids.
     * @param {Tracker} tracker The tracker to get the names from.
     * @param {number} id Id to get the name for.
     * @param {number} maxSize The maximal size of the returned string.
     * @returns {string[]} The list of names.
     */
    static GetAllNames(tracker: Tracker, id: number, maxSize?: number): string[];
    /**
     * If there was an id merge, return the id that this id was merged into. Otherwise return the same id.
     * @param {Tracker} tracker The tracker to get the id from.
     * @param {number} id Id to get the merged id for.
     * @returns {number} The merged id.
     */
    static GetIDReassignment(tracker: Tracker, id: number): number;
    /**
     * Get the number of ids similar to this id.
     * @param {Tracker} tracker The tracker to get the number of ids from.
     * @param {number} id Id to get the number of similar ids for.
     * @returns {number} The number of similar ids.
     */
    static GetSimilarIDCount(tracker: Tracker, id: number): number;
    /**
     * Get the similar ids to this id.
     * @param {Tracker} tracker The tracker to get the similar ids from.
     * @param {number} id Id to get similar ids for.
     * @returns {number[]} Array of ids.
     */
    static GetSimilarIDList(tracker: Tracker, id: number): number[];
    /**
     * Get the total number of ids stored in tracker memory.
     * @param {Tracker} tracker The tracker to get number of ids from.
     * @returns {number} The number of ids.
     */
    static GetTrackerIDsCount(tracker: Tracker): number;
    /**
     * Get all tracker ids.
     * @param {Tracker} tracker The tracker to get ids from.
     * @returns {number[]} Array of tracker ids.
     */
    static GetTrackerAllIDs(tracker: Tracker): number[];
    /**
     * Get the number of face ids for id.
     * @param {Tracker} tracker The tracker to get the number of face ids from.
     * @param {number} id Id to get the number of face ids for.
     * @returns {number} The number of face ids.
     */
    static GetTrackerFaceIDsCountForID(tracker: Tracker, id: number): number;
    /**
     * Get face ids for id.
     * @param {Tracker} tracker The tracker to get the face ids from.
     * @param {number} id Id to get face ids for.
     * @returns {number[]} Array of face ids.
     */
    static GetTrackerFaceIDsForID(tracker: Tracker, id: number): number[];
    /**
     * Get id for a face id.
     * @param {Tracker} tracker The tracker to get the id from.
     * @param {number} faceID Face id to get id for.
     * @returns {number} The id.
     */
    static GetTrackerIDByFaceID(tracker: Tracker, faceID: number): number;
    /**
     * Get the face template for face id.
     * @param {Tracker} tracker The tracker to get face template from.
     * @param {number} faceID Face id to get face template for.
     * @returns {FaceTemplate} The face template.
     */
    static GetTrackerFaceTemplate(tracker: Tracker, faceID: number): FaceTemplate;
    static GetTrackerFaceImage(tracker: Tracker, faceID: number): Image;
    /**
     * Get the face image for face id.
     * @param {Tracker} tracker The tracker to get face image from.
     * @param {number} faceID Face id to get face image for.
     * @returns {Image} The face image.
     */
    static SetTrackerFaceImage(tracker: Tracker, faceID: number, image: Image): void;
    /**
     * Delete the face image for face id.
     * @param {Tracker} tracker The tracker to delete face image from.
     * @param {number} faceID Face id to delete the face image for.
     * @returns {void}
     */
    static DeleteTrackerFaceImage(tracker: Tracker, faceID: number): void;
    /**
     * Create a new id in the tracker memory.
     * @param {Tracker} tracker The tracker to create a new id in.
     * @param {FaceTemplate} template Face template of the id.
     * @returns {number} The newly created id.
     */
    static TrackerCreateID(tracker: Tracker, template: FaceTemplate): TrackerID;
    /**
     * Add a new face template for id.
     * @param {Tracker} tracker The tracker to add a new face template to.
     * @param {number} id Id to add a new template for.
     * @param {FaceTemplate} template The template.
     * @returns {void}
     */
    static AddTrackerFaceTemplate(tracker: Tracker, id: number, template: FaceTemplate): void;
    /**
     * Delete face id.
     * @param {Tracker} tracker The tracker to delete the face id in.
     * @param {number} faceID Face id to delete.
     * @returns {void}
     */
    static DeleteTrackerFace(tracker: Tracker, faceID: number): void;
    /**
     * Get ids and their similarities for a face template.
     * @param {Tracker} tracker The tracker to match faces in.
     * @param {FaceTemplate} template The template to find similar ids for.
     * @param {number} threshold Matching similarity threshold for the returned ids.
     * @param {number} maxSize Maximal number of ids to return.
     * @returns {IDSimilarity[]} Array of ids and their similarities.
     */
    static TrackerMatchFaces(tracker: Tracker, template: FaceTemplate, threshold: number, maxSize?: number): IDSimilarity[];
    /**
     * Get facial attribute values (i.e. angles, liveness) for an id.
     * @template {FacialAttribute[]} A
     * @param {Tracker} tracker The tracker to get facial attributes in.
     * @param {number} id Id to get attributes for.
     * @param {A} attributes The attributes to get.
     * @returns {Object} An object with attribute values. The obejct keys depend on which attriutes are requested.
     */
    static GetTrackerFacialAttribute<A extends FacialAttribute[]>(tracker: Tracker, id: number, ...attributes: A): FlatType<FacialAttributesResults<A>>;
    /**
     * Get facial attribute values (i.e. angles, liveness) for an id.
     * @param {Tracker} tracker The tracker to get facial attributes in.
     * @param {number} id Id to get attributes for.
     * @param {string | string[]} attribute The attribute (or array of attributes) to get.
     * @param {string} maxSize The maximal size of the returned string.
     * @param {number} index Camera index (unused).
     * @returns {string} A string with the attribute values in key1=value1;key2=value2; format.
     */
    static GetTrackerFacialAttributeRaw(tracker: Tracker, id: number, attribute: string | string[], maxSize?: number, index?: number): string;
    /**
     * Detect facial attribute values (i.e. angles, liveness) using facial keypoints.
     * @template {FacialAttribute[]} A
     * @param {Image} image The image to get facial attribute form.
     * @param {Point[]} features Array of facial keypoints.
     * @param {A} attributes The attributes to detect.
     * @returns {Object} An object with attribute values. The obejct keys depend on which attriutes are requested.
     */
    static DetectFacialAttributeUsingFeatures<A extends FacialAttribute[]>(image: Image, features: Point[], ...attributes: A): FlatType<FacialAttributesResults<A>>;
    /**
     * Detect facial attribute values (i.e. angles, liveness) using facial keypoints.
     * @param {Image} image The image to get facial attribute form.
     * @param {Point[]} features Array of facial keypoints.
     * @param {string | string[]} attribute The attribute (or array of attributes) to detect.
     * @param {string} maxSize The maximal size of the returned string.
     * @returns {string} A string with the detected attribute values in key1=value1;key2=value2; format.
     */
    static DetectFacialAttributeUsingFeaturesRaw(image: Image, features: Point[], attribute: string | string[], maxSize?: number): string;
    /**
     * Get value from a key=value; string.
     * @param {string} values The string in key1=value1;key2=value2; format.
     * @param {string} value The key for the value to obtain.
     * @returns {number} The value.
     */
    static GetValueConfidence(values: string, value: FacialAttribute | TrackerFacialAttribute): number;
    /**
     * Set FSDK parameter.
     * @template {Parameter} P
     * @param {P} parameter The parameter to set.
     * @param {ParameterValueType<P>} value The value to set.
     * @returns {value}
     */
    static SetParameter<P extends Parameter>(parameter: P, value: ParameterValueType<P>): void;
    /**
     * Set FSDK parameters.
     * @template {Parameter} T Parameters to set.
     * @param {string | Parameters<T>} parameters Parameters to set. If string, must have the following format 'key1=value1;key2=value2;...'
     * @returns {number} Index of the parsing error position if any.
     */
    static SetParameters<T extends Parameter>(parameters: string | Parameters<T>): number;
    /**
     * Set an HTTP proxy to be used with an IP camera.
     * @param {string} address Proxy address.
     * @param {number} port Proxy port.
     * @param {string} username Proxy username.
     * @param {string} password Proxy password.
     * @returns {void}
     */
    static SetHTTPProxy(address: string, port: number, username: string, password: string): void;
    /**
     * Open IP video camera at a given URL.
     * @param {FlatType<VIDEOCOMPRESSIONTYPE>} comression Video compression to use.
     * @param {string} url Camera URL.
     * @param {string} username Camera access username.
     * @param {string} password Camera access password.
     * @param {number} timeout Connection timeout in seconds.
     * @returns {Camera} The IP video camera.
     */
    static OpenIPVideoCamera(comression: VIDEOCOMPRESSIONTYPE, url: string, username: string, password: string, timeout: number): Camera;
    /**
     * Initialize the capturing process.
     * @returns {void}
     */
    static InitializeCapturing(): void;
    /**
     * Finalize the capturing process.
     * @returns {void}
     */
    static FinalizeCapturing(): void;
    /**
     * Obtain the current camera frame image.
     * @param {Camera} camera The camera to grab the frame from.
     * @returns {Image} The frame image.
     */
    static GrabFrame(camera: Camera): Image;
    /**
     * Close the camera. Camera becomes invalid.
     * @param {Camera} camera The camera to close.
     * @returns {void}
     */
    static CloseVideoCamera(camera: Camera): void;
    /**
     * Initlize IBeta addon.
     * @returns {Promise<void>}
     */
    static InitializeIBeta(): Promise<void>;
}
//# sourceMappingURL=index.d.ts.map