#include "stdafx.h"
#include "resource.h"
#include "LuxandFaceSDK.h"

// Window procedure callback function for handling window messages
LRESULT CALLBACK	WndProc(HWND, UINT, WPARAM, LPARAM);

// Flag indicating whether the application should close
bool NeedClose;

// Array storing unique IDs for detected faces (up to 256 faces)
long long IDs[256];

// Current number of detected faces in the frame
long long faceCount = 0;

// Array storing rectangle coordinates for each detected face
RECT faceRects[256]; 

// Current X coordinate of mouse cursor
int MouseX = 0;
// Current Y coordinate of mouse cursor
int MouseY = 0;

// List of available camera names (wide character strings)
wchar_t ** CameraList;
// Number of available cameras in the system
int CameraCount = 0;

// List of available video formats for the selected camera
FSDK_VideoFormatInfo * VideoFormatList = 0;

// Number of available video formats
int VideoFormatCount = 0;

// Structure representing an RGB color with red, green, and blue components
struct RGBColor {
	unsigned char R = 0;
	unsigned char G = 0;
	unsigned char B = 0;
};

// Red pen used for drawing rectangles around faces (e.g., for non-live faces)
HPEN redPen;

// Green pen used for drawing rectangles around live faces
HPEN greenPen;

/**
 * Opens the camera and selects the highest resolution video format
 * @param cameraHandle Output parameter for camera handle
 * @param width Output parameter for video width
 * @param height Output parameter for video height
 * @param bitsPerPixel Output parameter for bits per pixel
 * @return 0 on success, -1 if no camera found, -2 if camera open failed
 */
int OpenCamera(int & cameraHandle, int & width, int & height, int & bitsPerPixel) {
	printf("Getting camera list...\n");

    if (0 == FSDK_GetCameraList(&CameraList, &CameraCount))
		for (int i = 0; i < CameraCount; i++) 
			wprintf(L"camera: %s\n", CameraList[i]);

	if (0 == CameraCount) {
		MessageBox(0, L"Please attach a camera", L"Error", MB_ICONERROR | MB_OK);
        return -1;
    }

	int CameraIdx = 0; // choose the first camera
	FSDK_GetVideoFormatList(CameraList[CameraIdx], &VideoFormatList, &VideoFormatCount);

	// choose the format with the highest resolution
	int formatIndex = 0; 
	int max_width = 0, max_height = 0;
	for (int i = 0; i < VideoFormatCount ; i++) {
		printf("format %d: %dx%d, %d bpp\n", i, VideoFormatList[i].Width, VideoFormatList[i].Height, VideoFormatList[i].BPP);
		if (VideoFormatList[i].Width > max_width && VideoFormatList[i].Height > max_height) {
			max_width = VideoFormatList[i].Width;
			max_height = VideoFormatList[i].Height;
			formatIndex = i;
		}
	}

	int VideoFormat = formatIndex;
	width = VideoFormatList[VideoFormat].Width;
	height = VideoFormatList[VideoFormat].Height;
	bitsPerPixel = VideoFormatList[VideoFormat].BPP; 

	printf("Choosing format %d: %dx%d, %d bpp\n", VideoFormat, width, height, bitsPerPixel);
	
	FSDK_SetVideoFormat(CameraList[0], VideoFormatList[VideoFormat]);

	printf("Trying to open the first camera...\n");
	cameraHandle = 0;
	if (FSDKE_OK != FSDK_OpenVideoCamera(CameraList[0], &cameraHandle)) { 
		MessageBox(0, L"Error opening the first camera", L"Error", MB_ICONERROR | MB_OK);
       return -2;
	} 
	return 0;
}

/**
 * Initializes the main application window with specified dimensions
 * @param width Width of the window content area
 * @param height Height of the window content area
 * @param dc Output parameter for device context handle
 * @param hwnd Output parameter for window handle
 * @param TextBox Output parameter for text box control handle
 * @return 0 on success
 */
int InitializeWindow(int width, int height, HDC & dc, HWND & hwnd, HWND & TextBox) {
	LOGBRUSH brush;
	brush.lbColor = RGB(230, 230, 230);
	brush.lbStyle = BS_SOLID;
	HBRUSH bBrush;
	bBrush=CreateBrushIndirect(&brush);
	WNDCLASSEX wcex;
	wcex.cbSize = sizeof(WNDCLASSEX);
	wcex.style			= 0;
	wcex.lpfnWndProc	= WndProc;
	wcex.cbClsExtra		= 0;
	wcex.cbWndExtra		= 0;
	wcex.hInstance		= 0;
	wcex.hIcon			= 0;
	wcex.hCursor		= LoadCursor(NULL, IDC_ARROW);
	wcex.hbrBackground	= bBrush;
	wcex.lpszMenuName	= 0;
	wcex.lpszClassName	= L"My Window Class";
	wcex.hIconSm		= 0;
	RegisterClassEx(&wcex);

	hwnd = CreateWindowEx(WS_EX_TOOLWINDOW, L"My Window Class", L"Passive Liveness", WS_SYSMENU, 0, 0, 0, 0, 0, 0, 0, 0); 
	dc = GetDC(hwnd);
	SetWindowPos(hwnd, 0, 0, 0, 6+width, 6+16+(height)+60, SWP_NOZORDER|SWP_NOMOVE);
	ShowWindow(hwnd, SW_SHOW);

	redPen = CreatePen(PS_SOLID, 1, RGB(255, 0, 0));
	greenPen = CreatePen(PS_SOLID, 1, RGB(0, 255, 0));

	HPEN hNPen = CreatePen(PS_SOLID, 1, RGB(0, 255, 0));
	HPEN hOPen = (HPEN)SelectObject(dc, hNPen);
	HBRUSH hOldBrush;
	HBRUSH hNewBrush;
	hNewBrush = (HBRUSH)GetStockObject(NULL_BRUSH);
	hOldBrush = (HBRUSH)SelectObject(dc, hNewBrush);

	UpdateWindow(hwnd);
	return 0;
}

/**
 * Determines which face rectangle (if any) the mouse cursor is currently over
 * Also changes the cursor to a hand icon when over a face
 * @return Index of the selected face rectangle, or -1 if none selected
 */
int GetSelectedRectangle() {
	int RectSelected = -1;
	for (int i = 0; i < faceCount; i++) 
		if (MouseX >= faceRects[i].left && MouseX <= faceRects[i].right && 
			MouseY >= faceRects[i].top && MouseY <= faceRects[i].bottom) 
		{
			RectSelected = i;
			break;
		}
	
	if (RectSelected >= 0) {
		HCURSOR Cur = LoadCursor(NULL, IDC_HAND);
		SetCursor(Cur);
	}
	return RectSelected;
}

/**
 * Displays an image from FaceSDK image handle to the device context
 * Converts the image to HBITMAP and draws it on the DC
 * @param dc Device context handle to draw on
 * @param imageHandle FaceSDK image handle to display
 */
void DisplayImage(HDC dc, HImage imageHandle) {	
	int width = 0; 
	int height = 0;
	FSDK_GetImageWidth(imageHandle, &width);
	FSDK_GetImageHeight(imageHandle, &height);
	HBITMAP hbitmapHandle; // to store the HBITMAP handle
	FSDK_SaveImageToHBitmap(imageHandle, &hbitmapHandle);
	DrawState(dc, NULL, NULL, (LPARAM)hbitmapHandle, NULL, 0, 0, width, height, DST_BITMAP | DSS_NORMAL);		
	DeleteObject(hbitmapHandle); // delete the HBITMAP object
}

/**
 * Displays text on the device context with specified color and position
 * @param dc Device context handle to draw on
 * @param color RGB color for the text
 * @param name Text string to display
 * @param r Rectangle defining the text display area
 */
void DisplayText(HDC dc, RGBColor color, char * name, RECT r) {
	SetBkMode(dc, TRANSPARENT);
	SetTextColor(dc, RGB(color.R, color.G, color.B));
	HFONT MyFont = CreateFont(25, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, L"Microsoft Sans Serif");
	SelectObject(dc, MyFont);
	DrawTextA(dc, name, -1, &r, DT_CENTER);
	DeleteObject(MyFont);
}

/**
 * Main entry point of the application
 * Initializes FaceSDK, opens camera, creates face tracker, and runs the main loop
 * @param argc Number of command line arguments
 * @param argv Array of command line argument strings
 * @return 0 on success, negative value on error
 */
int _tmain(int argc, _TCHAR* argv[])
{
    if (FSDKE_OK != FSDK_ActivateLibrary("INSERT THE LICENSE KEY HERE")) {
		MessageBox(0, L"Please insert the license key in the FSDK_ActivateLibrary function\n", L"Error activating FaceSDK", MB_ICONERROR | MB_OK);
        exit(-1);
    }

	FSDK_Initialize("");
	FSDK_InitializeCapturing();

	int cameraHandle = 0;
	int width, height, bitsPerPixel;
	int res = OpenCamera(cameraHandle, width, height, bitsPerPixel);
	if (res < 0) return res;

	// creating a Tracker
	HTracker tracker = 0;
	FSDK_CreateTracker(&tracker);

	int err = 0; // set realtime face detection parameters
	FSDK_SetTrackerMultipleParameters(tracker, "RecognizeFaces=false;HandleArbitraryRotations=false; DetermineFaceRotationAngle=false; InternalResizeWidth=100; FaceDetectionThreshold=5;", &err);

	FSDK_SetTrackerParameter(tracker, "DetectLiveness", "true");
	FSDK_SetTrackerParameter(tracker, "LivenessFramesCount", "6");
	FSDK_SetTrackerParameter(tracker, "SmoothAttributeLiveness", "true");

	HDC dc;
	HWND hwnd, TextBox;
	InitializeWindow(width, height, dc, hwnd, TextBox);
					
	MSG msg = {0};
	NeedClose = FALSE;

	while (msg.message != WM_QUIT) {
		HImage imageHandle;
		FSDK_GrabFrame(cameraHandle, &imageHandle); // receive a frame from the camera
		FSDK_FeedFrame(tracker, 0, imageHandle, &faceCount, IDs, sizeof(IDs));
		DisplayImage(dc, imageHandle);
	
		for (int i = 0; i < faceCount; i++) {
			TFacePosition facePosition;
			FSDK_GetTrackerFacePosition(tracker, 0, IDs[i], &facePosition);

			RGBColor rectColor {0, 255, 0};
			SelectObject(dc, greenPen);

			float liveness = 0.0f;
			char livenessAttribute[256];
			bool hasLivenessAttribute = false;
			if (FSDKE_OK == FSDK_GetTrackerFacialAttribute(tracker, 0, IDs[i], "Liveness", livenessAttribute, sizeof(livenessAttribute))) {
				if (FSDKE_OK == FSDK_GetValueConfidence(livenessAttribute, "Liveness", &liveness)) {
					hasLivenessAttribute = true;
					if (liveness < 0.5f) { // not live
						rectColor.R = 255;
						rectColor.G = 0;
						rectColor.B = 0;
						SelectObject(dc, redPen);
					}
				}
			}
			
			faceRects[i].left = facePosition.xc - (int)(facePosition.w*0.6);
			faceRects[i].top = min(height, facePosition.yc - (int)(facePosition.w*0.5));
			faceRects[i].right = facePosition.xc + (int)(facePosition.w*0.6);
			faceRects[i].bottom = min(height, facePosition.yc + (int)(facePosition.w*0.7));
			Rectangle(dc, faceRects[i].left, faceRects[i].top, faceRects[i].right, faceRects[i].bottom);

			if (hasLivenessAttribute) {
				RECT LivenessRect;
				LivenessRect.left = faceRects[i].left - width;
				LivenessRect.right = faceRects[i].right + width;
				LivenessRect.top = faceRects[i].top - 30;
				LivenessRect.bottom = faceRects[i].top;
				char livenessText[256];
				sprintf_s(livenessText, "Liveness: %.1f%%", liveness * 100.0f);
				DisplayText(dc, rectColor, livenessText, LivenessRect);
			}
		}
		
		if (PeekMessage(&msg, 0, 0, 0, PM_REMOVE)) {
			TranslateMessage(&msg);   
			DispatchMessage(&msg); 
			if ((msg.message == WM_KEYDOWN && msg.wParam == VK_ESCAPE) || NeedClose) 
				break;
		} 

		FSDK_FreeImage(imageHandle); // delete the FaceSDK image handle
	}

	ReleaseDC(hwnd, dc);

	FSDK_FreeTracker(tracker);

	if (FSDKE_OK != FSDK_CloseVideoCamera(cameraHandle)) {
		MessageBox(0, L"Error closing camera", L"Error", MB_ICONERROR | MB_OK);
        return -5;
	}

	FSDK_FreeVideoFormatList(VideoFormatList);
	FSDK_FreeCameraList(CameraList, CameraCount);

	FSDK_FinalizeCapturing();
	FSDK_Finalize();
	return 0;
}

/**
 * Window procedure callback for handling window messages
 * Processes mouse movement, mouse clicks, and window destruction events
 * @param hWnd Handle to the window
 * @param message Message identifier
 * @param wParam Additional message-specific information
 * @param lParam Additional message-specific information
 * @return Result of message processing
 */
LRESULT CALLBACK WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) {
		case WM_MOUSEMOVE:
			MouseX = LOWORD(lParam); 
			MouseY = HIWORD(lParam);			
			return (INT_PTR)TRUE;
	
		case WM_LBUTTONUP:
			return (INT_PTR)TRUE;	
	
		case WM_CTLCOLORSTATIC:
			SetBkMode((HDC)wParam, TRANSPARENT);
			return (int)CreateSolidBrush(RGB(230, 230, 230));

		case WM_DESTROY:
			NeedClose = TRUE;
			break;
	}
	return DefWindowProc(hWnd, message, wParam, lParam);
}