package com.luxand.passiveliveness;

import Luxand.FSDK;
import Luxand.FSDKCam;
import Luxand.FSDK.HImage;
import Luxand.FSDK.HTracker;
import Luxand.FSDKCam.HCamera;
import Luxand.FSDKCam.TCameras;
import Luxand.FSDK.FSDK_IMAGEMODE;
import Luxand.FSDKCam.FSDK_VideoFormats;

import java.awt.*;
import javax.swing.*;

import java.awt.font.TextLayout;
import java.awt.geom.Rectangle2D;
import java.util.concurrent.Semaphore;
import java.awt.font.FontRenderContext;

public class PassiveLivenessPanel extends JPanel {

    private static final int MAX_FACES = 5;
    private static final Stroke faceStroke = new BasicStroke(5);
    private static final Font textFont = new Font("Arial", Font.PLAIN, 25);

    private final HImage image = new HImage();
    private final HCamera camera = new HCamera();
    private final Image[] awtImage = new Image[2];
    private final HTracker tracker = new HTracker();
    private final Semaphore semaphore = new Semaphore(1);

    private final long[] facesCount = new long[1];
    private final long[] IDs = new long[MAX_FACES];
    private final Face[] faces = new Face[MAX_FACES];
    private final String[] value = new String[1];

    private double ratioX = 0;
    private double ratioY = 0;
    private final int[] imageWidth = new int[1];
    private final int[] imageHeight = new int[1];

    public PassiveLivenessPanel() {
        for (int i = 0; i < faces.length; ++i)
            faces[i] = new Face();
    }
    
    public void initialize() {
        Utils.checkErrorCode(() -> FSDK.CreateTracker(tracker), "FSDK_CreateTracker");

        int[] err = new int[1];
        Utils.checkErrorCode(() -> FSDK.SetTrackerMultipleParameters(tracker,
                "RecognizeFaces=false;" +
                "DetectFacialFeatures=true;" +
                "InternalResizeWidth=256;" +
                "HandleArbitraryRotations=false;" +
                "DetermineFaceRotationAngle=false;" +
                "FaceDetectionThreshold=5;" +
                "DetectLiveness=true;",
                err
        ), "FSDK_SetTrackerMultipleParameters");

        Utils.checkErrorCode(FSDKCam::InitializeCapturing, "FSDK_InitializeCapturing");

        final int[] count = new int[1];
        final TCameras cameras = new TCameras();
        Utils.checkErrorCode(() -> FSDKCam.GetCameraList(cameras, count), "FSDK_GetCameraList");
        if (cameras.cameras.length == 0) {
            JOptionPane.showMessageDialog(null, "No camera devices found", "FaceSDK Error", JOptionPane.ERROR_MESSAGE);
            System.exit(-1);
        }

        final String cameraName = cameras.cameras[0];
        final FSDK_VideoFormats videoFormats = new FSDK_VideoFormats();
        Utils.checkErrorCode(() -> FSDKCam.GetVideoFormatList(cameraName, videoFormats, count), "FSDK_GetVideoFormatList");
        Utils.checkErrorCode(() -> FSDKCam.SetVideoFormat(cameraName, videoFormats.formats[0]), "FSDK_SetVideoFormat");
        Utils.checkErrorCode(() -> FSDKCam.OpenVideoCamera(cameraName, camera), "FSDK_OpenCamera");

        new Thread(this::livenessDetection).start();
    }

    private void livenessDetection() {
        while (true) {
            Utils.checkErrorCode(() -> FSDKCam.GrabFrame(camera, image), "FSDK_GrabFrame");
            Utils.checkErrorCode(() -> FSDK.MirrorImage(image, true), "FSDK_MirrorImage");
            Utils.checkErrorCode(() -> FSDK.SaveImageToAWTImage(image, awtImage, FSDK_IMAGEMODE.FSDK_IMAGE_COLOR_24BIT), "FSDK_SaveToAWTImage");
            Utils.checkErrorCode(() -> FSDK.FeedFrame(tracker, 0, image, facesCount, IDs), "FSDK_FeedFrame");

            if (imageWidth[0] == 0) {
                Utils.checkErrorCode(() -> FSDK.GetImageWidth(image, imageWidth), "FSDK_GetImageWidth");
                Utils.checkErrorCode(() -> FSDK.GetImageHeight(image, imageHeight), "FSDK_GetImageHeight");
                SwingUtilities.getWindowAncestor(this).setSize(imageWidth[0], imageHeight[0]);
            }

            ratioX = (double)getWidth()  / imageWidth[0];
            ratioY = (double)getHeight() / imageHeight[0];

            Utils.checkErrorCode(() -> FSDK.FreeImage(image), "FSDK_FreeImage");

            semaphore.acquireUninterruptibly();
            awtImage[1] = awtImage[0];
            
            for (int i = 0; i < facesCount[0]; ++i) {
                faces[i].setID(IDs[i]);
                FSDK.GetTrackerFacePosition(tracker, 0, IDs[i], faces[i].getFacePositionReference());
                if (FSDK.GetTrackerFacialAttribute(tracker, 0, IDs[i], "Liveness", value, 1024) == FSDK.FSDKE_OK) {
                    FSDK.GetValueConfidence(value[0], "Liveness", faces[i].getSimilarityReference());
                }
                else
                    faces[i].setSimilarity(-1);
            }
            
            if (facesCount[0] < faces.length)
                faces[(int)facesCount[0]].setID(-1);

            semaphore.release();
            
            SwingUtilities.invokeLater(this::repaint);
        }
    }

    private void drawString(final String value, final Face.Frame frame, final Color color, final Font font, final Graphics2D graphics) {
        final FontRenderContext frc = graphics.getFontRenderContext();
        final TextLayout layout = new TextLayout(value, font, frc);
        final Rectangle2D bounds = layout.getBounds();
        
        final int x = (int)((frame.x + frame.width / 2) * ratioX - bounds.getWidth() / 2);
        final int y = (int)((frame.y + frame.height) * ratioY + 3 * bounds.getHeight());
        
        graphics.setFont(font);
        
        graphics.setColor(color);
        graphics.drawString(value, x, y);
    }

    @Override
    protected void paintComponent(Graphics g) {
        super.paintComponent(g);
        semaphore.acquireUninterruptibly();
        g.drawImage(awtImage[1], 0, 0, getWidth(), getHeight(), this);

        final Graphics2D graphics = (Graphics2D)g;
        
        for (Face face : faces) {
            if (face.getID() < 0)
                break;
            
            graphics.setStroke(faceStroke);
            
            final Face.Frame frame = face.getFrame();
            final float similarity = face.getSimilarity();
            if (similarity > 0.5)
                drawString("Live", frame, Color.green, textFont, graphics);
            else if (similarity > 0)
                drawString("Fake", frame, Color.red, textFont, graphics);
            else
                graphics.setColor(Color.blue);
            graphics.drawRect(frame.x, frame.y, frame.width, frame.height);
        }
        semaphore.release();
    }
}
