//
//  RecognitionViewController.swift
//  LiveRecognition
//
//  Copyright © 2025 Luxand, Inc. All rights reserved.
//

import Foundation
import UIKit
import AVKit

class RecognitionViewController: UIViewController, RecognitionCameraDelegate, UIAlertViewDelegate {
    struct DetectFaceParams {
        var buffer: UnsafeMutablePointer<UInt8>
        var width: Int32
        var height: Int32
        var scanline: Int32
        var ratio: Float32
    }
    
    struct FaceRectangle {
        var x1: Int32
        var x2: Int32
        var y1: Int32
        var y2: Int32
    }
    
    let MAX_FACES = 5
    let MAX_NAME_LEN = 1024
    let ATTRIB_VERTEX = 0
    let ATTRIB_TEXTUREPOSITON = 1
    let NUM_ATTRIBUTES = 2
    let HELP_TEXT = "Just tap any detected face and name it. The app will recognize this face further. For best results, hold the device at arm's length. You may slowly rotate the head for the app to memorize you at multiple views. The app can memorize several persons. If a face is not recognized, tap and name it again. The SDK is available for mobile developers: www.luxand.com/facesdk"
    
    var camera: RecognitionCamera?
    var cameraPosition: AVCaptureDevice.Position = .front
    var screenForDisplay: UIScreen?
    var directDisplayProgram: GLuint = 0
    var videoFrameTexture: GLuint = 0
    var rawPositionPixels: GLubyte = 0
    var rotating: Bool = false
    var videoStarted: Bool = false
    var processingImage: Bool = false
    var toolbar: UIToolbar?
    var orientation: UIInterfaceOrientation = UIInterfaceOrientation.unknown
    
    // face processing
    var closing: Bool = false
    var enteredNameLock: NSLock = NSLock()
    var enteredName: String = String()
    var enteredNameChanged: Bool = false
    var namedFaceID: Int64 = -1
    var trackingRects = [CALayer]()
    var nameLabels = [CATextLayer]()
    var faceDataLock: NSLock = NSLock()
    var faces = [FaceRectangle]()
    var nameDataLock: NSLock = NSLock()
    var names = NSMutableArray()
    var statuses = NSMutableArray()
    var IDs = [Int64]()
    var faceTouchedLock: NSLock = NSLock()
    var faceTouched: Bool = false
    var indexOfTouchedFace: Int = -1
    var idOfTouchedFace: Int64 = -1
    var currentTouchPoint: CGPoint = CGPoint()
    var clearTracker: Bool = false
    var clearTrackerLock: NSLock = NSLock()
    var tracker: HTracker = 0
    var templatePath: String = String()
    
    var shiftX: Int32 = 0
    var shiftY: Int32 = 0
    
    required init?(coder aDecoder: NSCoder) {
        super.init(coder: aDecoder)
    }
    
    init(screen: UIScreen) {
        super.init(nibName: nil, bundle: nil)
        
        screenForDisplay = screen
        rotating = false
        processingImage = false
        trackingRects.removeAll()
        nameLabels.removeAll()
        faces.removeAll()
        names.removeAllObjects()
        IDs.removeAll()
        for _ in 1...MAX_FACES {
            trackingRects.append(CALayer())
            nameLabels.append(CATextLayer())
            faces.append(FaceRectangle(x1: 0, x2: 0, y1: 0, y2: 0))
            names.add(String())
            statuses.add(String())
            IDs.append(-1)
        }
        
        // Load tracker memory
        let homeDir = NSHomeDirectory()
        let filePath = "Documents/Memory70.dat"
        if (homeDir.last == "/") {
            templatePath = "\(homeDir)\(filePath)"
        } else {
            templatePath = "\(homeDir)/\(filePath)"
        }
        if (FSDKE_OK != FSDK_LoadTrackerMemoryFromFile(&tracker, (templatePath as NSString).utf8String)) {
            FSDK_CreateTracker(&tracker)
        }
        resetTrackerParameters()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        UIApplication.shared.isIdleTimerDisabled = true
    }
    
    @objc
    func clearAction(sender: UIBarButtonItem) {
        let alert = UIAlertView(title: "", message: "Are you sure to clear the memory?", delegate: self, cancelButtonTitle: "Cancel", otherButtonTitles: "Ok")
        alert.tag = 0xC
        alert.show()
    }
    
    @objc
    func helpAction(sender: UIBarButtonItem) {
        let alert = UIAlertView(title: "Luxand Face Recognition", message: HELP_TEXT, delegate: nil, cancelButtonTitle: "Ok")
        alert.show()
    }
    
    func recreateCamera() {
        camera?.videoPreviewLayer?.removeFromSuperlayer()
        camera = RecognitionCamera(position: cameraPosition);
        camera?.delegate = self
        self.cameraHasConnected()
                
        camera?.videoPreviewLayer?.frame = self.view.bounds
        
        shiftX = 0
        shiftY = 0
        let w: Int32 = Int32(self.view.bounds.width)
        let h: Int32 = Int32(self.view.bounds.height)
        let camw: Int32 = camera?.width ?? 0
        let camh: Int32 = camera?.height ?? 0
        if (w > h) {
            shiftX = (w - (camw * h) / camh) / 2
        } else {
            shiftY = (h - (camw * w) / camh) / 2
        }
        
        switch(orientation) {
            case .landscapeRight: camera?.videoPreviewLayer?.connection?.videoOrientation = .landscapeRight
            case .landscapeLeft: camera?.videoPreviewLayer?.connection?.videoOrientation = .landscapeLeft
            case .portraitUpsideDown: camera?.videoPreviewLayer?.connection?.videoOrientation = .portraitUpsideDown
            default: camera?.videoPreviewLayer?.connection?.videoOrientation = .portrait
        }
        self.view.layer.insertSublayer((camera?.videoPreviewLayer)!, at: 0)
    }
    
    @objc
    func switchCameraAction(sender: UIBarButtonItem) {
        if (cameraPosition == .front) {
            cameraPosition = .back
        } else {
            cameraPosition = .front
        }
        
        recreateCamera()
        
        view.bringSubview(toFront: toolbar!)
    }
    
    
    func alertView(_ alertView: UIAlertView, clickedButtonAt buttonIndex: Int) {
        if (alertView.tag == 0xC) { // Clear memory
            if (buttonIndex == 1) {
                clearTrackerLock.lock()
                clearTracker = true
                clearTrackerLock.unlock()
            }
        } else if (alertView.tag == 0xF) { // enter name for the Face
            faceTouchedLock.lock()
            if (buttonIndex == 0 || idOfTouchedFace == -1) {
                faceTouched = false
                faceTouchedLock.unlock()
                return
            }
            let name = alertView.textField(at: 0)?.text
            let truncName = String(name!.prefix(MAX_NAME_LEN))
            enteredNameLock.lock()
            namedFaceID = idOfTouchedFace
            enteredName = truncName
            enteredNameChanged = true
            // immediately display the name
            let i = Int(indexOfTouchedFace)
            if (i >= 0) {
                nameDataLock.lock()
                names.replaceObject(at: i, with: truncName)
                nameDataLock.unlock()
            }
            faceTouched = false
            enteredNameLock.unlock()
            faceTouchedLock.unlock()
        }
    }
    
    
    // touch handling
    
    func pointInRectangle(point_x: Int32, point_y: Int32, rect_x1: Int32, rect_y1: Int32, rect_x2: Int32, rect_y2: Int32) -> Bool {
        return (point_x >= rect_x1) && (point_x <= rect_x2) && (point_y >= rect_y1) && (point_y <= rect_y2)
    }
    
    override func touchesBegan(_ touches: Set<UITouch>, with event: UIEvent?) {
        currentTouchPoint = touches.first!.location(in: self.view)
        let x = Int32(currentTouchPoint.x)
        let y = Int32(currentTouchPoint.y)
        
        faceTouchedLock.lock()
        idOfTouchedFace = -1
        
        faceDataLock.lock()
        for i in 0...MAX_FACES-1 {
            if (pointInRectangle(point_x: x, point_y: y, rect_x1: faces[i].x1, rect_y1: faces[i].y1, rect_x2: faces[i].x2, rect_y2: faces[i].y2+30)) {
                indexOfTouchedFace = i
                idOfTouchedFace = IDs[i]
                break
            }
        }
        faceDataLock.unlock()
        
        if (idOfTouchedFace >= 0) {
            faceTouched = true
            faceTouchedLock.unlock()
            
            let alert = UIAlertView(title: "", message: "Enter person's name", delegate: self, cancelButtonTitle: "Cancel", otherButtonTitles: "Ok")
            alert.alertViewStyle = UIAlertViewStyle.plainTextInput
            alert.tag = 0xF
            alert.show()
        } else {
            faceTouchedLock.unlock()
        }
    }
    
    
    // device rotation support
    
    override func willRotate(to toInterfaceOrientation: UIInterfaceOrientation, duration: TimeInterval) {
        rotating = true
        camera?.videoPreviewLayer?.isHidden = true
        for i in 0...MAX_FACES-1 {
            trackingRects[i].isHidden = true
        }
        toolbar?.isHidden = true
    }
    
    override func didRotate(from fromInterfaceOrientation: UIInterfaceOrientation) {
        for i in 0...MAX_FACES-1 {
            trackingRects[i].isHidden = false
        }
        rotating = false
    }
    
    override func loadView() {
        let mainScreenFrame: CGRect? = UIScreen.main.bounds
        let primaryView: UIView = UIView(frame: mainScreenFrame!)
        view = primaryView
        
        // Set up the toolbar at the bottom of the screen
        toolbar = UIToolbar()
        toolbar?.barStyle = UIBarStyle.black;
        let clearItem = UIBarButtonItem(barButtonSystemItem: UIBarButtonSystemItem.trash, target: self, action: #selector(clearAction))
        let flexibleSpace = UIBarButtonItem(barButtonSystemItem: UIBarButtonSystemItem.flexibleSpace, target: nil, action: nil)
        let switchCameraItem = UIBarButtonItem(title: "Camera", style: UIBarButtonItemStyle.plain, target: self, action: #selector(switchCameraAction))
        let helpItem = UIBarButtonItem(title: "?", style: UIBarButtonItemStyle.plain, target: self, action: #selector(helpAction))
        toolbar?.items = [clearItem, flexibleSpace, switchCameraItem, helpItem]
        toolbar?.sizeToFit()
        let toolbarHeight = toolbar?.frame.size.height
        let mainViewBounds = view.bounds
        toolbar?.frame = CGRect(x: mainViewBounds.minX, y: mainViewBounds.minY + mainViewBounds.height - toolbarHeight!, width: mainViewBounds.width, height: toolbarHeight!)
        view.addSubview(toolbar!)
        view.bringSubview(toFront: toolbar!)
        
        for i in 0...MAX_FACES-1 {
            trackingRects[i].bounds = CGRect(x: 0.0, y: 0.0, width: 0.0, height: 0.0)
            trackingRects[i].cornerRadius = 0.0
            trackingRects[i].borderColor = UIColor.blue.cgColor
            trackingRects[i].borderWidth = 2.0
            trackingRects[i].position = CGPoint(x: 100, y: 100)
            trackingRects[i].opacity = 0.0
            trackingRects[i].anchorPoint = CGPoint(x: 0, y: 0) //for position to be the top-left corner
            nameLabels[i].fontSize = 20
            nameLabels[i].frame = CGRect(x: 10.0, y: 10.0, width: 200.0, height: 40.0)
            nameLabels[i].string = "Tap to name"
            nameLabels[i].foregroundColor = UIColor.green.cgColor
            nameLabels[i].alignmentMode = kCAAlignmentCenter
            trackingRects[i].addSublayer(nameLabels[i])
            
            // Disable animations for move and resize (otherwise trackingRect will jump)
            trackingRects[i].actions = ["sublayer":NSNull(),"position":NSNull(),"bounds":NSNull()]
            nameLabels[i].actions = ["sublayer":NSNull(),"position":NSNull(),"bounds":NSNull()]
        }
                
        camera = RecognitionCamera(position: cameraPosition);
        camera?.delegate = self
        camera?.videoPreviewLayer?.frame = self.view.bounds
        self.view.layer.addSublayer((camera?.videoPreviewLayer)!)
        
        for i in 0...MAX_FACES-1 {
            self.view.layer.addSublayer(trackingRects[i])
        }

    }
    
    func screenSizeOrientationIndependent() -> CGSize {
        let screenSize = UIScreen.main.bounds.size
        return CGSize(width: min(screenSize.width, screenSize.height), height: max(screenSize.width, screenSize.height))
    }
    
    func relocateSubviewsForOrientationChange() {
        // Toolbar re-alignment
        let toolbarHeight = toolbar?.frame.size.height
        let mainViewBounds = view.bounds
        toolbar?.frame = CGRect(x: mainViewBounds.minX, y: mainViewBounds.minY + mainViewBounds.height - toolbarHeight!, width: mainViewBounds.width, height: toolbarHeight!)
        view.addSubview(toolbar!)

        recreateCamera()
        
        view.bringSubview(toFront: toolbar!)
        toolbar?.isHidden = false
    }
    
    func processNewCameraFrame(cameraFrame: CVImageBuffer) {
        if (rotating) {
            return; //not updating GLView on rotating animation (it looks ugly)
        }
        
        CVPixelBufferLockBaseAddress(cameraFrame, CVPixelBufferLockFlags(rawValue: 0))
        let bufferHeight = Int(CVPixelBufferGetHeight(cameraFrame))
        let bufferWidth = Int(CVPixelBufferGetWidth(cameraFrame))
        
        // Create a new texture from the camera frame data, draw it (calling drawFrame)
        drawFrame()
        
        
        faceTouchedLock.lock()
        let faceTouchedVal = faceTouched
        faceTouchedLock.unlock()
        
        if (processingImage == false && !faceTouchedVal) {
            if (closing) {
                return
            }
            processingImage = true
            
            // Copy camera frame to buffer
            
            let scanline = CVPixelBufferGetBytesPerRow(cameraFrame)
            let buffer = UnsafeMutablePointer<UInt8>.allocate(capacity: scanline * bufferHeight)
            memcpy(buffer, CVPixelBufferGetBaseAddress(cameraFrame), scanline * bufferHeight)
            
            var ratio = Float(0)
            if #available(iOS 13.0, *) {
                if let windowScene = UIApplication.shared.connectedScenes.first as? UIWindowScene {
                    let interfaceOrientation = windowScene.interfaceOrientation
                    
                    if (interfaceOrientation == .unknown || interfaceOrientation == .portrait || interfaceOrientation == .portraitUpsideDown) {
                        ratio = Float(self.view.bounds.size.width) / Float(bufferHeight)
                    } else {
                        ratio = Float(self.view.bounds.size.height) / Float(bufferHeight)
                    }
                }
            } else {
                let orientation = UIApplication.shared.statusBarOrientation
                if (orientation == UIInterfaceOrientation.unknown || orientation == UIInterfaceOrientation.portrait || orientation == UIInterfaceOrientation.portraitUpsideDown) {
                    ratio = Float(self.view.bounds.size.width) / Float(bufferHeight)
                } else {
                    ratio = Float(self.view.bounds.size.height) / Float(bufferHeight)
                }
            }
            
            let args = DetectFaceParams(buffer: buffer, width: Int32(bufferWidth), height: Int32(bufferHeight), scanline: Int32(scanline), ratio: ratio)
            
            DispatchQueue.global(qos: .background).async {
                self.processImage(args: args)
            }
        }

        CVPixelBufferUnlockBaseAddress(cameraFrame, CVPixelBufferLockFlags(rawValue: 0))
    }
    
    
    
    func drawFrame() {
        if #available(iOS 13.0, *) {
            if let windowScene = UIApplication.shared.connectedScenes.first as? UIWindowScene {
                let orientation = windowScene.interfaceOrientation
                if orientation != self.orientation {
                    self.orientation = orientation
                    relocateSubviewsForOrientationChange()
                }
            }
        } else {
            let orientation: UIInterfaceOrientation = UIApplication.shared.statusBarOrientation
            if (orientation != self.orientation) {
                self.orientation = orientation
                relocateSubviewsForOrientationChange()
            }
        }
        
        // For some reason that's mandatory for new devices
        CATransaction.begin()
        CATransaction.setDisableActions(true)
        
        // Setting bounds and position of trackingRect using data received from FSDK_DetectFace
        // need to disable animations because we can show incorrect (old) name for a moment in result
        
        faceDataLock.lock()
        nameDataLock.lock()
        
        for i in 0..<MAX_FACES {
            let s = (statuses[i] as! String)
            let name = (names[i] as! String)
            
            let isLive = s != "" && s.contains("Live")
            let isFake = s != "" && s.contains("Fake")
            let hasLiveness = isLive || isFake
            
            if (name != "") {
                nameLabels[i].string = "\(name)  \(s)"
            } else {
                nameLabels[i].string = "Tap to name \(s)"
            }
            
            if (hasLiveness) {
                nameLabels[i].foregroundColor = isLive ? UIColor.green.cgColor : UIColor.red.cgColor
            } else {
                nameLabels[i].foregroundColor = name != "" ? UIColor.blue.cgColor : UIColor.green.cgColor
            }
        }
        
        for i in 0..<MAX_FACES {
            if (faces[i].x2 > 0) { // have face
                nameLabels[i].frame = CGRect(x: 10.0, y: Double(faces[i].y2 - faces[i].y1) + 10.0, width: Double(faces[i].x2 - faces[i].x1) - 20.0, height: 40.0)
                trackingRects[i].position = CGPoint(x: Int(faces[i].x1), y: Int(faces[i].y1))
                trackingRects[i].bounds = CGRect(x: 0.0, y: 0.0, width: Double(faces[i].x2 - faces[i].x1), height: Double(faces[i].y2 - faces[i].y1))
                trackingRects[i].opacity = 1.0
            } else { // no face
                trackingRects[i].opacity = 0.0
            }
        }
        
        nameDataLock.unlock()
        faceDataLock.unlock()
        
        CATransaction.commit()
        videoStarted = true
    }
    
    func cameraHasConnected() {
    }
    
    override func viewDidLoad() {
        super.viewDidLoad()
        // Do any additional setup after loading the view, typically from a nib.
    }
    
    override func didReceiveMemoryWarning() {
        super.didReceiveMemoryWarning()
        // Dispose of any resources that can be recreated.
    }
    
    
    // Face detection and recognition
    
    func processImage(args: DetectFaceParams) {
        if (closing) {
            processingImage = false
            return
        }
        
        // Cleaning tracker memory, if the button was pressed
        
        clearTrackerLock.lock()
        if (clearTracker) {
            FSDK_ClearTracker(tracker)
            resetTrackerParameters()
            clearTracker = false
        }
        clearTrackerLock.unlock()
        
        // Reading buffer parameters
        let width: Int32 = args.width
        let height: Int32 = args.height
        let scanline: Int32 = args.scanline
        let ratio: Float32 = args.ratio
        
        
        // Converting BGRA to RGBA
        CSwapChannels13I(args.buffer, scanline, width, height, 4)
        /*
         var p1line = args.buffer
         var p2line = args.buffer + 2
         var y = 0
         while(y < height) {
         var p1 = p1line
         var p2 = p2line
         p1line += Int(scanline)
         p2line += Int(scanline)
         var x = 0
         while(x < width) {
         let tmp: UInt8 = p1.pointee
         p1.pointee = p2.pointee
         p2.pointee = tmp
         p1 += 4
         p2 += 4
         x += 1
         }
         y += 1
         }
         */
        
        
        var image: HImage = 0
        var res = FSDK_LoadImageFromBuffer(&image, args.buffer, width, height, scanline, FSDK_IMAGE_COLOR_32BIT)
        
        args.buffer.deallocate()
        
        if (res != FSDKE_OK) {
            print("FSDK_LoadImageFromBuffer failed with \(res)")
            processingImage = false
            return
        }
        
        var derotatedImage: HImage = 0
        res = FSDK_CreateEmptyImage(&derotatedImage)
        if (res != FSDKE_OK) {
            FSDK_FreeImage(image)
            processingImage = false
            return
        }
        
        
        if #available(iOS 13.0, *) {
            if let windowScene = UIApplication.shared.connectedScenes.first as? UIWindowScene {
                let df_orientation = windowScene.interfaceOrientation
                if df_orientation == .unknown || df_orientation == .portrait {
                    res = FSDK_RotateImage90(image, 1, derotatedImage)
                } else if df_orientation == .portraitUpsideDown {
                    res = FSDK_RotateImage90(image, -1, derotatedImage)
                } else if (df_orientation == .landscapeLeft && self.cameraPosition == .front) || (df_orientation == .landscapeRight && self.cameraPosition == .back) {
                    res = FSDK_RotateImage90(image, 0, derotatedImage)
                } else if (df_orientation == .landscapeRight && self.cameraPosition == .front) || (df_orientation == .landscapeLeft && self.cameraPosition == .back) {
                    res = FSDK_RotateImage90(image, 2, derotatedImage)
                }
            }
        } else {
            let df_orientation = UIApplication.shared.statusBarOrientation;
            if (df_orientation == UIInterfaceOrientation.unknown || df_orientation == UIInterfaceOrientation.portrait) {
                res = FSDK_RotateImage90(image, 1, derotatedImage)
            } else if (df_orientation == UIInterfaceOrientation.portraitUpsideDown) {
                res = FSDK_RotateImage90(image, -1, derotatedImage)
            } else if ((df_orientation == .landscapeLeft && cameraPosition == .front) || (df_orientation == .landscapeRight && cameraPosition == .back)) {
                res = FSDK_RotateImage90(image, 0, derotatedImage)
            } else if ((df_orientation == .landscapeRight && cameraPosition == .front) || (df_orientation == .landscapeLeft && cameraPosition == .back)) {
                res = FSDK_RotateImage90(image, 2, derotatedImage)
            }
        }
        
        if (res != FSDKE_OK) {
            FSDK_FreeImage(image)
            FSDK_FreeImage(derotatedImage)
            processingImage = false
            return
        }
        
        if (cameraPosition == .front) {
            res = FSDK_MirrorImage_uchar(derotatedImage, 1)
            if (res != FSDKE_OK) {
                FSDK_FreeImage(image)
                FSDK_FreeImage(derotatedImage)
                processingImage = false
                return
            }
        }
        
        
        // Passing entered name to FaceSDK
        enteredNameLock.lock()
        if (enteredNameChanged) {
            if (FSDKE_OK == FSDK_LockID(tracker, namedFaceID)) {
                FSDK_SetName(tracker, namedFaceID, (enteredName as NSString).utf8String)
                if (enteredName == "") {
                    FSDK_PurgeID(tracker, namedFaceID)
                }
                FSDK_UnlockID(tracker, namedFaceID)
            }
        }
        enteredName = ""
        enteredNameChanged = false
        enteredNameLock.unlock()
        
        
        // Passing frame to FaceSDK, reading face coordinates and names
        var count: Int64 = 0
        FSDK_FeedFrame(tracker, 0, derotatedImage, &count, &IDs, Int64(IDs.count * Int64.bitWidth/8))
        
        faceDataLock.lock()
        nameDataLock.lock()
        for i in 0..<MAX_FACES {
            faces[i].x1 = 0
            faces[i].x2 = 0
            faces[i].y1 = 0
            faces[i].y2 = 0
            names.replaceObject(at: i, with: String.init())
            statuses.replaceObject(at: i, with: "")
        }
        for i in 0..<Int(count) {
            let allNames = UnsafeMutablePointer<Int8>.allocate(capacity: MAX_NAME_LEN+1)
            allNames.initialize(repeating: 0, count: MAX_NAME_LEN+1)
            FSDK_GetAllNames(tracker, IDs[i], allNames, Int64(MAX_NAME_LEN))
            names.replaceObject(at: i, with: String.init(cString: allNames))
            allNames.deallocate()
            
            let eyesPtr = UnsafeMutablePointer<FSDK_Features>.allocate(capacity: 1)
            var eyes : FSDK_Features = eyesPtr.pointee
            
            FSDK_GetTrackerEyes(tracker, 0, IDs[i], &eyes)
            
            let face = getFaceFrame(eyes: &eyes)
            
            faces[i].x1 = Int32(Float32(face.x1) * ratio) + shiftX
            faces[i].x2 = Int32(Float32(face.x2) * ratio) + shiftX
            faces[i].y1 = Int32(Float32(face.y1) * ratio) + shiftY
            faces[i].y2 = Int32(Float32(face.y2) * ratio) + shiftY
            
            eyesPtr.deallocate()
            
            let attributes = UnsafeMutablePointer<Int8>.allocate(capacity: 256)
                                    
            let res: Int32 = FSDK_GetTrackerFacialAttribute(tracker, 0, IDs[i], "Liveness", attributes, 256);
            if (res == FSDKE_OK) {
                let livenessThreshold: Float = 0.5;
                var livenessValue: Float = 0.0

                if (FSDK_GetValueConfidence(attributes, "Liveness", &livenessValue) == FSDKE_OK) {
                    var status: String = "";
                    if (livenessValue > livenessThreshold) {
                        status = "Live (\(livenessValue))"
                    } else {
                        status = "Fake (\(livenessValue))"
                    }
                    statuses.replaceObject(at: i, with: status)
                }
            }
            
            
        }
        nameDataLock.unlock()
        faceDataLock.unlock()
        
        
        FSDK_FreeImage(image)
        FSDK_FreeImage(derotatedImage)
        processingImage = false
    }
    
    func getFaceFrame(eyes: inout FSDK_Features) -> FaceRectangle {
        let u1: Double = Double(eyes.0.x)
        let v1: Double = Double(eyes.0.y)
        let u2: Double = Double(eyes.1.x)
        let v2: Double = Double(eyes.1.y)
        let xc: Double = (u1 + u2)/2
        let yc: Double = (v1 + v2)/2
        let w = sqrt((u2 - u1)*(u2 - u1) + (v2 - v1)*(v2 - v1))
        let x1 = Int32(xc - w * 1.6 * 0.9)
        let y1 = Int32(yc - w * 1.1 * 0.9)
        var x2 = Int32(xc + w * 1.6 * 0.9)
        var y2 = Int32(yc + w * 2.1 * 0.9)
        if (x2 - x1 > y2 - y1) {
            x2 = x1 + y2 - y1
        } else {
            y2 = y1 + x2 - x1
        }
        return FaceRectangle(x1: x1, x2: x2, y1: y1, y2: y2)
    }
    
    func resetTrackerParameters() {
        var errpos: Int32 = 0
        FSDK_SetTrackerMultipleParameters(tracker, ("ContinuousVideoFeed=true;FacialFeatureJitterSuppression=0;RecognitionPrecision=1;Threshold=0.992;Threshold2=0.9995;ThresholdFeed=0.97;MemoryLimit=2000;HandleArbitraryRotations=false;DetermineFaceRotationAngle=false;InternalResizeWidth=128;FaceDetectionThreshold=3" as NSString).utf8String, &errpos)
        
        FSDK_SetTrackerParameter(tracker, "DetectLiveness", "true"); // enable liveness
        FSDK_SetTrackerParameter(tracker, "SmoothAttributeLiveness", "false"); // use smooth minimum function for liveness values
        FSDK_SetTrackerParameter(tracker, "AttributeLivenessSmoothingAlpha", "1"); // smooth minimum parameter, 0 -> mean, inf -> min
        FSDK_SetTrackerParameter(tracker, "LivenessFramesCount", "1"); // minimal number of frames required to output liveness attribute
    }
    
}
