//
//  RecognitionViewController.swift
//  LiveRecognition
//
//  Copyright © 2025 Luxand, Inc. All rights reserved.
//

import Foundation
import UIKit
import AVKit

class RecognitionViewController: UIViewController, RecognitionCameraDelegate, UIAlertViewDelegate {
    struct DetectFaceParams {
        var buffer: UnsafeMutablePointer<UInt8>
        var width: Int32
        var height: Int32
        var scanline: Int32
        var ratio: Float32
    }
    
    struct FaceRectangle {
        var x1: Int32
        var x2: Int32
        var y1: Int32
        var y2: Int32
    }
    
    let MAX_FACES = 5
    let HELP_TEXT = "Luxand Age Gender Expression recognition\n\nThis sample demonstrates how to use Luxand FaceSDK to detect age, gender, and expression.\n\nThe SDK is available for mobile developers: www.luxand.com/facesdk"
    
    var camera: RecognitionCamera?
    var cameraPosition: AVCaptureDevice.Position = .front
    var rotating: Bool = false
    var videoStarted: Bool = false
    var processingImage: Bool = false
    var toolbar: UIToolbar?
    var orientation: UIInterfaceOrientation = UIInterfaceOrientation.unknown
    
    // face processing
    var closing: Bool = false
    var trackingRects = [CALayer]()
    var nameLabels = [CATextLayer]()
    var faceDataLock: NSLock = NSLock()
    var faces = [FaceRectangle]()
    var nameDataLock: NSLock = NSLock()
    var statuses = NSMutableArray()
    var IDs = [Int64]()
    var tracker: HTracker = 0
    
    var shiftX: Int32 = 0
    var shiftY: Int32 = 0
    
    required init?(coder aDecoder: NSCoder) {
        super.init(coder: aDecoder)
    }
    
    init(screen: UIScreen) {
        super.init(nibName: nil, bundle: nil)
        
        rotating = false
        processingImage = false
        trackingRects.removeAll()
        nameLabels.removeAll()
        faces.removeAll()
        IDs.removeAll()

        for _ in 1...MAX_FACES {
            trackingRects.append(CALayer())
            nameLabels.append(CATextLayer())
            faces.append(FaceRectangle(x1: 0, x2: 0, y1: 0, y2: 0))
            statuses.add(String())
            IDs.append(-1)
        }
        
        FSDK_CreateTracker(&tracker)
        resetTrackerParameters()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        UIApplication.shared.isIdleTimerDisabled = true
    }
        
    @objc
    func helpAction(sender: UIBarButtonItem) {
        let alert = UIAlertView(title: "Luxand Face Recognition", message: HELP_TEXT, delegate: nil, cancelButtonTitle: "Ok")
        alert.show()
    }
    
    func recreateCamera() {
        camera?.videoPreviewLayer?.removeFromSuperlayer()
        camera = RecognitionCamera(position: cameraPosition);
        camera?.delegate = self
        self.cameraHasConnected()
                
        camera?.videoPreviewLayer?.frame = self.view.bounds
        
        shiftX = 0
        shiftY = 0
        let w: Int32 = Int32(self.view.bounds.width)
        let h: Int32 = Int32(self.view.bounds.height)
        let camw: Int32 = camera?.width ?? 0
        let camh: Int32 = camera?.height ?? 0
        if (w > h) {
            shiftX = (w - (camw * h) / camh) / 2
        } else {
            shiftY = (h - (camw * w) / camh) / 2
        }
        
        switch(orientation) {
            case .landscapeRight: camera?.videoPreviewLayer?.connection?.videoOrientation = .landscapeRight
            case .landscapeLeft: camera?.videoPreviewLayer?.connection?.videoOrientation = .landscapeLeft
            case .portraitUpsideDown: camera?.videoPreviewLayer?.connection?.videoOrientation = .portraitUpsideDown
            default: camera?.videoPreviewLayer?.connection?.videoOrientation = .portrait
        }
        self.view.layer.insertSublayer((camera?.videoPreviewLayer)!, at: 0)
    }
    
    @objc
    func switchCameraAction(sender: UIBarButtonItem) {
        if (cameraPosition == .front) {
            cameraPosition = .back
        } else {
            cameraPosition = .front
        }
        
        recreateCamera()
        
        view.bringSubview(toFront: toolbar!)
    }
    
    // device rotation support
    
    override func willRotate(to toInterfaceOrientation: UIInterfaceOrientation, duration: TimeInterval) {
        rotating = true
        camera?.videoPreviewLayer?.isHidden = true
        for i in 0...MAX_FACES-1 {
            trackingRects[i].isHidden = true
        }
        toolbar?.isHidden = true
    }
    
    override func didRotate(from fromInterfaceOrientation: UIInterfaceOrientation) {
        for i in 0...MAX_FACES-1 {
            trackingRects[i].isHidden = false
        }
        rotating = false
    }
    
    override func loadView() {
        let mainScreenFrame: CGRect? = UIScreen.main.bounds
        let primaryView: UIView = UIView(frame: mainScreenFrame!)
        view = primaryView
        
        // Set up the toolbar at the bottom of the screen
        toolbar = UIToolbar()
        toolbar?.barStyle = UIBarStyle.black;
        let flexibleSpace = UIBarButtonItem(barButtonSystemItem: UIBarButtonSystemItem.flexibleSpace, target: nil, action: nil)
        let switchCameraItem = UIBarButtonItem(title: "Camera", style: UIBarButtonItemStyle.plain, target: self, action: #selector(switchCameraAction))
        let helpItem = UIBarButtonItem(title: "?", style: UIBarButtonItemStyle.plain, target: self, action: #selector(helpAction))
        toolbar?.items = [flexibleSpace, switchCameraItem, helpItem]
        toolbar?.sizeToFit()
        let toolbarHeight = toolbar?.frame.size.height
        let mainViewBounds = view.bounds
        toolbar?.frame = CGRect(x: mainViewBounds.minX, y: mainViewBounds.minY + mainViewBounds.height - toolbarHeight!, width: mainViewBounds.width, height: toolbarHeight!)
        view.addSubview(toolbar!)
        view.bringSubview(toFront: toolbar!)
        
        for i in 0...MAX_FACES-1 {
            trackingRects[i].bounds = CGRect(x: 0.0, y: 0.0, width: 0.0, height: 0.0)
            trackingRects[i].cornerRadius = 0.0
            trackingRects[i].borderColor = UIColor.blue.cgColor
            trackingRects[i].borderWidth = 2.0
            trackingRects[i].position = CGPoint(x: 100, y: 100)
            trackingRects[i].opacity = 0.0
            trackingRects[i].anchorPoint = CGPoint(x: 0, y: 0) //for position to be the top-left corner
            nameLabels[i].fontSize = 20
            nameLabels[i].frame = CGRect(x: 10.0, y: 10.0, width: 200.0, height: 40.0)
            nameLabels[i].string = ""
            nameLabels[i].foregroundColor = UIColor.green.cgColor
            nameLabels[i].alignmentMode = kCAAlignmentCenter
            trackingRects[i].addSublayer(nameLabels[i])
            
            // Disable animations for move and resize (otherwise trackingRect will jump)
            trackingRects[i].actions = ["sublayer":NSNull(),"position":NSNull(),"bounds":NSNull()]
            nameLabels[i].actions = ["sublayer":NSNull(),"position":NSNull(),"bounds":NSNull()]
        }
                
        camera = RecognitionCamera(position: cameraPosition);
        camera?.delegate = self
        camera?.videoPreviewLayer?.frame = self.view.bounds
        self.view.layer.addSublayer((camera?.videoPreviewLayer)!)
        
        for i in 0...MAX_FACES-1 {
            self.view.layer.addSublayer(trackingRects[i])
        }

    }
    
    func screenSizeOrientationIndependent() -> CGSize {
        let screenSize = UIScreen.main.bounds.size
        return CGSize(width: min(screenSize.width, screenSize.height), height: max(screenSize.width, screenSize.height))
    }
    
    func relocateSubviewsForOrientationChange() {
        // Toolbar re-alignment
        let toolbarHeight = toolbar?.frame.size.height
        let mainViewBounds = view.bounds
        toolbar?.frame = CGRect(x: mainViewBounds.minX, y: mainViewBounds.minY + mainViewBounds.height - toolbarHeight!, width: mainViewBounds.width, height: toolbarHeight!)
        view.addSubview(toolbar!)

        recreateCamera()
        
        view.bringSubview(toFront: toolbar!)
        toolbar?.isHidden = false
    }
    
    func processNewCameraFrame(cameraFrame: CVImageBuffer) {
        if (rotating) {
            return; //not updating GLView on rotating animation (it looks ugly)
        }
        
        CVPixelBufferLockBaseAddress(cameraFrame, CVPixelBufferLockFlags(rawValue: 0))
        let bufferHeight = Int(CVPixelBufferGetHeight(cameraFrame))
        let bufferWidth = Int(CVPixelBufferGetWidth(cameraFrame))
        
        // Create a new texture from the camera frame data, draw it (calling drawFrame)
        drawFrame()
                
        if (processingImage == false) {
            if (closing) {
                return
            }
            processingImage = true
            
            // Copy camera frame to buffer
            
            let scanline = CVPixelBufferGetBytesPerRow(cameraFrame)
            let buffer = UnsafeMutablePointer<UInt8>.allocate(capacity: scanline * bufferHeight)
            memcpy(buffer, CVPixelBufferGetBaseAddress(cameraFrame), scanline * bufferHeight)
            
            var ratio = Float(0)
            if #available(iOS 13.0, *) {
                if let windowScene = UIApplication.shared.connectedScenes.first as? UIWindowScene {
                    let interfaceOrientation = windowScene.interfaceOrientation
                    
                    if (interfaceOrientation == .unknown || interfaceOrientation == .portrait || interfaceOrientation == .portraitUpsideDown) {
                        ratio = Float(self.view.bounds.size.width) / Float(bufferHeight)
                    } else {
                        ratio = Float(self.view.bounds.size.height) / Float(bufferHeight)
                    }
                }
            } else {
                let orientation = UIApplication.shared.statusBarOrientation
                if (orientation == UIInterfaceOrientation.unknown || orientation == UIInterfaceOrientation.portrait || orientation == UIInterfaceOrientation.portraitUpsideDown) {
                    ratio = Float(self.view.bounds.size.width) / Float(bufferHeight)
                } else {
                    ratio = Float(self.view.bounds.size.height) / Float(bufferHeight)
                }
            }
            
            let args = DetectFaceParams(buffer: buffer, width: Int32(bufferWidth), height: Int32(bufferHeight), scanline: Int32(scanline), ratio: ratio)
            
            DispatchQueue.global(qos: .background).async {
                self.processImage(args: args)
            }
        }

        CVPixelBufferUnlockBaseAddress(cameraFrame, CVPixelBufferLockFlags(rawValue: 0))
    }
    
    
    
    func drawFrame() {
        if #available(iOS 13.0, *) {
            if let windowScene = UIApplication.shared.connectedScenes.first as? UIWindowScene {
                let orientation = windowScene.interfaceOrientation
                if orientation != self.orientation {
                    self.orientation = orientation
                    relocateSubviewsForOrientationChange()
                }
            }
        } else {
            let orientation: UIInterfaceOrientation = UIApplication.shared.statusBarOrientation
            if (orientation != self.orientation) {
                self.orientation = orientation
                relocateSubviewsForOrientationChange()
            }
        }
        
        // For some reason that's mandatory for new devices
        CATransaction.begin()
        CATransaction.setDisableActions(true)
        
        // Setting bounds and position of trackingRect using data received from FSDK_DetectFace
        // need to disable animations because we can show incorrect (old) name for a moment in result
        
        faceDataLock.lock()
        nameDataLock.lock()
        
        for i in 0..<MAX_FACES {
            let s = (statuses[i] as! String)            
            nameLabels[i].string = s
            nameLabels[i].foregroundColor = UIColor.green.cgColor
        }
        
        for i in 0..<MAX_FACES {
            if (faces[i].x2 > 0) { // have face
                nameLabels[i].frame = CGRect(x: 10.0, y: Double(faces[i].y2 - faces[i].y1) + 10.0, width: Double(faces[i].x2 - faces[i].x1) - 20.0, height: 40.0)
                trackingRects[i].position = CGPoint(x: Int(faces[i].x1), y: Int(faces[i].y1))
                trackingRects[i].bounds = CGRect(x: 0.0, y: 0.0, width: Double(faces[i].x2 - faces[i].x1), height: Double(faces[i].y2 - faces[i].y1))
                trackingRects[i].opacity = 1.0
            } else { // no face
                trackingRects[i].opacity = 0.0
            }
        }
        
        nameDataLock.unlock()
        faceDataLock.unlock()
        
        CATransaction.commit()
        videoStarted = true
    }
    
    func cameraHasConnected() {
    }
    
    override func viewDidLoad() {
        super.viewDidLoad()
        // Do any additional setup after loading the view, typically from a nib.
    }
    
    override func didReceiveMemoryWarning() {
        super.didReceiveMemoryWarning()
        // Dispose of any resources that can be recreated.
    }
    
    
    // Face detection and recognition
    
    func processImage(args: DetectFaceParams) {
        if (closing) {
            processingImage = false
            return
        }
                
        // Reading buffer parameters
        let width: Int32 = args.width
        let height: Int32 = args.height
        let scanline: Int32 = args.scanline
        let ratio: Float32 = args.ratio
        
        // Converting BGRA to RGBA
        CSwapChannels13I(args.buffer, scanline, width, height, 4)
        
        var image: HImage = 0
        var res = FSDK_LoadImageFromBuffer(&image, args.buffer, width, height, scanline, FSDK_IMAGE_COLOR_32BIT)
        
        args.buffer.deallocate()
        
        if (res != FSDKE_OK) {
            print("FSDK_LoadImageFromBuffer failed with \(res)")
            processingImage = false
            return
        }
        
        var derotatedImage: HImage = 0
        res = FSDK_CreateEmptyImage(&derotatedImage)
        if (res != FSDKE_OK) {
            FSDK_FreeImage(image)
            processingImage = false
            return
        }
        
        
        if #available(iOS 13.0, *) {
            if let windowScene = UIApplication.shared.connectedScenes.first as? UIWindowScene {
                let df_orientation = windowScene.interfaceOrientation
                if df_orientation == .unknown || df_orientation == .portrait {
                    res = FSDK_RotateImage90(image, 1, derotatedImage)
                } else if df_orientation == .portraitUpsideDown {
                    res = FSDK_RotateImage90(image, -1, derotatedImage)
                } else if (df_orientation == .landscapeLeft && self.cameraPosition == .front) || (df_orientation == .landscapeRight && self.cameraPosition == .back) {
                    res = FSDK_RotateImage90(image, 0, derotatedImage)
                } else if (df_orientation == .landscapeRight && self.cameraPosition == .front) || (df_orientation == .landscapeLeft && self.cameraPosition == .back) {
                    res = FSDK_RotateImage90(image, 2, derotatedImage)
                }
            }
        } else {
            let df_orientation = UIApplication.shared.statusBarOrientation;
            if (df_orientation == UIInterfaceOrientation.unknown || df_orientation == UIInterfaceOrientation.portrait) {
                res = FSDK_RotateImage90(image, 1, derotatedImage)
            } else if (df_orientation == UIInterfaceOrientation.portraitUpsideDown) {
                res = FSDK_RotateImage90(image, -1, derotatedImage)
            } else if ((df_orientation == .landscapeLeft && cameraPosition == .front) || (df_orientation == .landscapeRight && cameraPosition == .back)) {
                res = FSDK_RotateImage90(image, 0, derotatedImage)
            } else if ((df_orientation == .landscapeRight && cameraPosition == .front) || (df_orientation == .landscapeLeft && cameraPosition == .back)) {
                res = FSDK_RotateImage90(image, 2, derotatedImage)
            }
        }
        
        if (res != FSDKE_OK) {
            FSDK_FreeImage(image)
            FSDK_FreeImage(derotatedImage)
            processingImage = false
            return
        }
        
        if (cameraPosition == .front) {
            res = FSDK_MirrorImage_uchar(derotatedImage, 1)
            if (res != FSDKE_OK) {
                FSDK_FreeImage(image)
                FSDK_FreeImage(derotatedImage)
                processingImage = false
                return
            }
        }
        
        // Passing frame to FaceSDK, reading face coordinates and names
        var count: Int64 = 0
        FSDK_FeedFrame(tracker, 0, derotatedImage, &count, &IDs, Int64(IDs.count * Int64.bitWidth/8))
        
        faceDataLock.lock()
        nameDataLock.lock()
        for i in 0..<MAX_FACES {
            faces[i].x1 = 0
            faces[i].x2 = 0
            faces[i].y1 = 0
            faces[i].y2 = 0
            statuses.replaceObject(at: i, with: "")
        }
        for i in 0..<Int(count) {

            let eyesPtr = UnsafeMutablePointer<FSDK_Features>.allocate(capacity: 1)
            var eyes : FSDK_Features = eyesPtr.pointee
            
            FSDK_GetTrackerEyes(tracker, 0, IDs[i], &eyes)
            
            let face = getFaceFrame(eyes: &eyes)
            
            faces[i].x1 = Int32(Float32(face.x1) * ratio) + shiftX
            faces[i].x2 = Int32(Float32(face.x2) * ratio) + shiftX
            faces[i].y1 = Int32(Float32(face.y1) * ratio) + shiftY
            faces[i].y2 = Int32(Float32(face.y2) * ratio) + shiftY
            
            eyesPtr.deallocate()

            let ageAttr = UnsafeMutablePointer<Int8>.allocate(capacity: 256)
            let genderAttr = UnsafeMutablePointer<Int8>.allocate(capacity: 256)
            let expressionAttr = UnsafeMutablePointer<Int8>.allocate(capacity: 256)

            var resAge: Int32 = FSDK_GetTrackerFacialAttribute(tracker, 0, IDs[i], "Age", ageAttr, 256);
            var resGender: Int32 = FSDK_GetTrackerFacialAttribute(tracker, 0, IDs[i], "Gender", genderAttr, 256);
            var resExpression: Int32 = FSDK_GetTrackerFacialAttribute(tracker, 0, IDs[i], "Expression", expressionAttr, 256);

            var ageString: String = ""
            var genderString: String = ""
            var expressionString: String = ""

            if (resAge == FSDKE_OK) {
                var ageValue: Float = 0.0
                if (FSDK_GetValueConfidence(ageAttr, "Age", &ageValue) == FSDKE_OK) {
                    ageString = String(format: "Age: %.0f", ageValue)
                }
            }

            if (resGender == FSDKE_OK) {
                var maleValue: Float = 0.0
                var femaleValue: Float = 0.0
                if ((FSDK_GetValueConfidence(genderAttr, "Male", &maleValue) == FSDKE_OK) && (FSDK_GetValueConfidence(genderAttr, "Female", &femaleValue) == FSDKE_OK)) 
                {
                    var genderConfidence: Float = maleValue > femaleValue ? maleValue : femaleValue
                    genderString = maleValue > femaleValue ? "Gender: Male" : "Gender: Female"
                    genderString += String(format: " (%.0f%%)", genderConfidence * 100)
                }
            }

            if (resExpression == FSDKE_OK) {
                var smileValue: Float = 0.0
                var eyesOpenValue: Float = 0.0
                if ((FSDK_GetValueConfidence(expressionAttr, "Smile", &smileValue) == FSDKE_OK) && (FSDK_GetValueConfidence(expressionAttr, "EyesOpen", &eyesOpenValue) == FSDKE_OK)) 
                {
                    expressionString = String(format: "Smile: %.0f%%, EyesOpen: %.0f%%", smileValue * 100, eyesOpenValue * 100)
                }
            }

            statuses.replaceObject(at: i, with: ageString + " " + genderString + " " + expressionString)

            ageAttr.deallocate()
            genderAttr.deallocate()
            expressionAttr.deallocate()
        }
        nameDataLock.unlock()
        faceDataLock.unlock()
        
        
        FSDK_FreeImage(image)
        FSDK_FreeImage(derotatedImage)
        processingImage = false
    }
    
    func getFaceFrame(eyes: inout FSDK_Features) -> FaceRectangle {
        let u1: Double = Double(eyes.0.x)
        let v1: Double = Double(eyes.0.y)
        let u2: Double = Double(eyes.1.x)
        let v2: Double = Double(eyes.1.y)
        let xc: Double = (u1 + u2)/2
        let yc: Double = (v1 + v2)/2
        let w = sqrt((u2 - u1)*(u2 - u1) + (v2 - v1)*(v2 - v1))
        let x1 = Int32(xc - w * 1.6 * 0.9)
        let y1 = Int32(yc - w * 1.1 * 0.9)
        var x2 = Int32(xc + w * 1.6 * 0.9)
        var y2 = Int32(yc + w * 2.1 * 0.9)
        if (x2 - x1 > y2 - y1) {
            x2 = x1 + y2 - y1
        } else {
            y2 = y1 + x2 - x1
        }
        return FaceRectangle(x1: x1, x2: x2, y1: y1, y2: y2)
    }
    
    func resetTrackerParameters() {
        var errpos: Int32 = 0
        FSDK_SetTrackerMultipleParameters(tracker, ("RecognizeFaces=false;DetectAge=true;DetectGender=true;DetectExpression=true;ContinuousVideoFeed=true;ThresholdFeed=0.97;MemoryLimit=2000;HandleArbitraryRotations=false;DetermineFaceRotationAngle=false;InternalResizeWidth=128;FaceDetectionThreshold=3" as NSString).utf8String, &errpos)
    }
    
}
