//
//  FaceLocator.swift
//  LivenessRecognition
//
//  Copyright © 2025 Luxand, Inc. All rights reserved.
//

import Foundation

class FaceLocator {
    class Result {
        public let frame: (Int32, Int32, Int32, Int32)
        public let text: String
        public let status: Bool
        
        init(_ status: Bool, _ text: String, _ frame: (Int32, Int32, Int32, Int32)) {
            self.status = status
            self.text = text
            self.frame = frame
        }
    }
    
    private class Point {
        public var x: Float = 0
        public var y: Float = 0
    }
    
    private class FaceRectangle {
        public var x1: Float = 0
        public var y1: Float = 0
        public var x2: Float = 0
        public var y2: Float = 0
        
        public func set(_ w: Float, _ h: Float) {
            x1 = -w / 2
            y1 = -h / 2
            x2 = w / 2
            y2 = h / 2
        }
    }
    
    private enum Command: CaseIterable {
        case TURN_LEFT
        case TURN_RIGHT
        case LOOK_STRAIGHT
        case LOOK_UP
        case LOOK_DOWN
        case SMILE
        
        static func random<G: RandomNumberGenerator>(using generator: inout G) -> Command {
                return Command.allCases.randomElement(using: &generator)!
            }
        
        static func random() -> Command {
            var g = SystemRandomNumberGenerator()
            return Command.random(using: &g)
        }
    }
    
    private let LEFT_RIGHT: Set = [ Command.TURN_LEFT, Command.TURN_RIGHT ]
    private let UP_DOWN: Set = [ Command.LOOK_DOWN, Command.LOOK_UP ]
  
    private let LEFT_EYE: Array<Int> = [ Int(FSDKP_LEFT_EYE),
                                         Int(FSDKP_LEFT_EYE_INNER_CORNER),
                                         Int(FSDKP_LEFT_EYE_OUTER_CORNER),
                                         Int(FSDKP_LEFT_EYE_LOWER_LINE1),
                                         Int(FSDKP_LEFT_EYE_LOWER_LINE2),
                                         Int(FSDKP_LEFT_EYE_LOWER_LINE3),
                                         Int(FSDKP_LEFT_EYE_UPPER_LINE1),
                                         Int(FSDKP_LEFT_EYE_UPPER_LINE2),
                                         Int(FSDKP_LEFT_EYE_UPPER_LINE3),
                                         Int(FSDKP_LEFT_EYE_RIGHT_IRIS_CORNER),
                                         Int(FSDKP_LEFT_EYE_LEFT_IRIS_CORNER) ]
    
    private let RIGHT_EYE: Array<Int> = [ Int(FSDKP_RIGHT_EYE),
                                          Int(FSDKP_RIGHT_EYE_INNER_CORNER),
                                          Int(FSDKP_RIGHT_EYE_OUTER_CORNER),
                                          Int(FSDKP_RIGHT_EYE_LOWER_LINE1),
                                          Int(FSDKP_RIGHT_EYE_LOWER_LINE2),
                                          Int(FSDKP_RIGHT_EYE_LOWER_LINE3),
                                          Int(FSDKP_RIGHT_EYE_UPPER_LINE1),
                                          Int(FSDKP_RIGHT_EYE_UPPER_LINE2),
                                          Int(FSDKP_RIGHT_EYE_UPPER_LINE3),
                                          Int(FSDKP_RIGHT_EYE_RIGHT_IRIS_CORNER),
                                          Int(FSDKP_RIGHT_EYE_LEFT_IRIS_CORNER) ]

    private var faceId: CLongLong
    private var commandId: Int = 0
    private var countdown: Int = 35

    private var ratio: Float
    private var angle: Double = 0

    private var frame = FaceRectangle()

    private var live = false

    private var center = Point()

    private var lpf: LowPassFilter? = nil
    private var pan = LowPassFilter()
    private var tilt = LowPassFilter()

    private var tracker: HTracker

    private var commands = Array<Command>()
    private var state: Set = [ Command.LOOK_STRAIGHT ]

    private var active = false

    private let commandNames: [Command : String] = [
        Command.TURN_LEFT : "Turn Left",
        Command.TURN_RIGHT : "Turn Right",
        Command.LOOK_STRAIGHT : "Look straight",
        Command.LOOK_UP : "Look up",
        Command.LOOK_DOWN : "Look down",
        Command.SMILE : "Make a smile",
    ]

    
    init(_ faceId: CLongLong, _ tracker: HTracker, _ ratio: Float) {
        self.faceId = faceId
        self.tracker = tracker
        self.ratio = ratio
    }
    
    public func doesIntersect(_ other: FaceLocator) -> Bool {
        return !(
                frame.x1 >= other.frame.x2 ||
                        frame.x2 < other.frame.x1 ||
                        frame.y1 >= other.frame.y2 ||
                        frame.y2 < other.frame.y1
        )
    }
    
    public func isInside(_ x: Double, _ y: Double) -> Bool {
        let x = x - Double(center.x) * Double(ratio)
        let y = y - Double(center.y) * Double(ratio)

        let a = angle * Double.pi / 180
        let xx = x * cos(a) + y * sin(a)
        let yy = x * sin(a) - y * cos(a)

        let res = pow(xx / Double(frame.x1) / Double(ratio), 2) + pow(yy / Double(frame.y1) / Double(ratio), 2) <= 1
        return res
    }

    public func isActive() -> Bool {
        return active
    }

    public func setActive(_ active: Bool) {
        self.active = active
        if (!active) {
            return
        }
        
        live = false
        commandId = 0
        
        commands = [ Command.LOOK_STRAIGHT ]
        
        while (commands.count < 6) {
            let cmd = Command.random()
            if (commands.last != cmd) {
                commands.append(cmd)
            }
        }
        
        if (!commands.contains(Command.SMILE)) {
            commands.insert(Command.SMILE, at: Int.random(in: 0..<commands.count))
        }
    }

    private func approveCommand(_ command: Command) -> Bool {
        if (!state.contains(command)) {
            return false
        }

        if (command == Command.SMILE) {
            return true
        }

        return !state.contains(Command.SMILE)
    }

    public func updateState() {
        var attributes = UnsafeMutablePointer<Int8>.allocate(capacity: 256)
        if (FSDK_GetTrackerFacialAttribute(tracker, 0, faceId, "Expression", attributes, 256) != FSDKE_OK) {
            attributes.deallocate()
            return;
        }
        var result = String(cString: attributes)
        attributes.deallocate()
        
        var smile: Double = 0

        for pair in result.components(separatedBy: ";") {
            let values = pair.components(separatedBy: "=")
            if (values[0].lowercased() == "smile") {
                smile = Double(values[1]) ?? 0
                break
            }
        }

        var pan: Float = 0
        var tilt: Float = 0

        attributes = UnsafeMutablePointer<Int8>.allocate(capacity: 256)
        if (FSDK_GetTrackerFacialAttribute(tracker, 0, faceId, "Angles", attributes, 256) != FSDKE_OK) {
            attributes.deallocate()
            return
        }
        result = String(cString: attributes)
        attributes.deallocate()

        for pair in result.components(separatedBy: ";") {
            let values = pair.components(separatedBy: "=")

            if (values[0].lowercased() == "pan") {
                pan = self.pan.pass(Float(values[1]) ?? 0)
                continue
            }

            if (values[0].lowercased() == "tilt") {
                tilt = self.tilt.pass(Float(values[1]) ?? 0)
                continue
            }
        }

        var st: Command? = nil

        let HOR_CONF_LEVEL_LOW: Float = 2
        let HOR_CONF_LEVEL_HIGH: Float = 17

        if (abs(pan) < HOR_CONF_LEVEL_LOW) {
            st = Command.LOOK_STRAIGHT
        } else if (abs(pan) > HOR_CONF_LEVEL_HIGH) {
            st = pan > 0 ? Command.TURN_LEFT : Command.TURN_RIGHT; // swapped for mirrored feed from front camera
        }

        if (st != nil) {
            LEFT_RIGHT.forEach { state.remove($0) }
            state.insert(st!)
        }

        let UP_CONF_LEVEL_LOW: Float = 2
        let UP_CONF_LEVEL_HIGH: Float = 5
        let DOWN_CONF_LEVEL_LOW: Float = -2
        let DOWN_CONF_LEVEL_HIGH: Float = -5

        st = nil

        if (tilt > UP_CONF_LEVEL_HIGH) {
            st = Command.LOOK_UP
        } else if (tilt < DOWN_CONF_LEVEL_HIGH) {
            st = Command.LOOK_DOWN
        } else if (tilt > DOWN_CONF_LEVEL_LOW && tilt < UP_CONF_LEVEL_LOW) {
            st = Command.LOOK_STRAIGHT
        }

        if (st != nil) {
            UP_DOWN.forEach { state.remove($0) }
            state.insert(st!)
        }

        if (state.contains(Command.LOOK_UP) || state.contains(Command.LOOK_DOWN) ||
                state.contains(Command.TURN_LEFT) || state.contains(Command.TURN_RIGHT)) {
            state.remove(Command.LOOK_STRAIGHT)
        }

        let SMILE_CONF_LEVEL_LOW: Double = 0.3
        let SMILE_CONF_LEVEL_HIGH: Double = 0.6

        if (smile < SMILE_CONF_LEVEL_LOW) {
            state.remove(Command.SMILE)
        } else if (smile > SMILE_CONF_LEVEL_HIGH) {
            state.insert(Command.SMILE)
        }

        if (active && !live) {
            if (approveCommand(commands[commandId])) {
                commandId += 1
                if (commandId >= commands.count) {
                    live = true
                }
            }
        }
    }

    private func dotCenter(_ points: Array<TPoint>) -> TPoint  {
        var result = TPoint()
        result.x = 0
        result.y = 0
        for point in points {
            result.x += point.x
            result.y += point.y
        }
        result.x /= Int32(points.count)
        result.y /= Int32(points.count)
        return result
    }

    private func getPoints(_ features: FSDK_Features, _ indices: Array<Int>) -> Array<TPoint> {
        var pFeatures = features
        
        let featuresArray = [TPoint](UnsafeBufferPointer(start: &(pFeatures.0), count: MemoryLayout.size(ofValue: features)))
        
        var result = Array<TPoint>()
        for i in indices {
            result.append(featuresArray[i])
        }
        return result
    }

    public func run() -> Result {
        let featuresPtr = UnsafeMutablePointer<FSDK_Features>.allocate(capacity: 1)
        var features: FSDK_Features = featuresPtr.pointee
        if (FSDK_GetTrackerFacialFeatures(tracker, 0, faceId, &features) != FSDKE_OK) {
            countdown -= 1;

            if (countdown <= 8) {
                frame.x1 *= 0.95
                frame.y1 *= 0.95
                frame.x2 *= 0.95
                frame.y2 *= 0.95
            }

            //drawShape(canvas);
            featuresPtr.deallocate()
            return Result(countdown > 0, "", getFaceFrame())
        }

        if (lpf == nil) {
            lpf = LowPassFilter()
        }

        let left = dotCenter(getPoints(features, LEFT_EYE))
        let right = dotCenter(getPoints(features, RIGHT_EYE))

        let w = lpf?.pass((Float(right.x - left.x)) * 2.8) ?? 0
        let h = w * 1.4

        center.x = (Float(right.x + left.x)) / 2.0
        center.y = (Float(right.y + left.y)) / 2.0 + w * 0.05
        angle = atan2(Double(right.y - left.y), Double(right.x - left.x)) * 180 / Double.pi

        frame.set(w, h)

        if (!active) {
            featuresPtr.deallocate()
            return Result(true, "", getFaceFrame());
        }

        var caption = live ? "LIVE!" : "";
        
        if (isActive()) {
            if (commandId >= commands.count) {
                caption = "LIVE!"
            } else {
                caption = "\(commandNames[commands[commandId]] ?? "Error") (\(commandId+1) of \(commands.count))"
            }
        }
            
        updateState()
        countdown = 35

        featuresPtr.deallocate()
        return Result(true, caption, getFaceFrame())
    }
    
    private func getFaceFrame() -> (Int32, Int32, Int32, Int32) {
        let left = (center.x + frame.x1) * ratio
        let top = (center.y + frame.y1) * ratio
        let right = (center.x + frame.x2) * ratio
        let bottom = (center.y + frame.y2) * ratio
        return (Int32(left), Int32(top), Int32(right), Int32(bottom))
    }
}
