// LivenessRecognition.cpp : This file contains the 'main' function. Program execution begins and ends there.
//

#include <iostream>
#include <vector>
#include <algorithm>
#include <ctime>
#include <unordered_map>
#include "windows.h"
#include "gdiplus.h"
#pragma comment(lib,"gdiplus.lib")
#include "LuxandFaceSDK.h"

using namespace std;

// Number of commands the user should perform for liveness check
const int NUM_OF_COMMANDS = 6; // the number of commands the user should perform for liveness check
// Number of frames to keep tracking a face after it's been missed
const int FRAMES_TO_KEEP_MISSED_FACE = 5;

// Horizontal rotation confidence levels (minimum and maximum thresholds)
int HOR_CONF_LEVEL[2] = {3, 18};
// Upward tilt confidence levels (minimum and maximum thresholds)
int UP_CONF_LEVEL[2] = {2, 5};
// Downward tilt confidence levels (minimum and maximum thresholds)
int DOWN_CONF_LEVEL[2] = {-2, -5};
// Smile detection confidence levels (minimum and maximum thresholds)
float SMILE_CONF_LEVEL[2] = {0.3f, 0.6f};

// Handle to the video camera device
int cameraHandle = -1;
// Video format information structure
FSDK_VideoFormatInfo videoFormat;
// Face tracker handle
HTracker tracker;

// Font size for text rendering
float FONT_SIZE = 14.0f;
// GDI+ font object for rendering text
Gdiplus::Font * font = 0;

// Handle to the main application window
HWND hWnd = 0;
// Flag indicating whether the application should exit
bool need_to_exit = false;

// GDI+ initialization token
ULONG_PTR gdip_token;

// Pi constant for angle calculations
#define PI 3.1415926f

// Array of facial feature points that define the left eye region
int LEFT_EYE_SET[] =  {FSDKP_LEFT_EYE, FSDKP_LEFT_EYE_INNER_CORNER, FSDKP_LEFT_EYE_OUTER_CORNER,
	FSDKP_LEFT_EYE_LOWER_LINE1, FSDKP_LEFT_EYE_LOWER_LINE2, FSDKP_LEFT_EYE_LOWER_LINE3,
	FSDKP_LEFT_EYE_UPPER_LINE1, FSDKP_LEFT_EYE_UPPER_LINE2, FSDKP_LEFT_EYE_UPPER_LINE3,
    FSDKP_LEFT_EYE_RIGHT_IRIS_CORNER, FSDKP_LEFT_EYE_LEFT_IRIS_CORNER};

// Array of facial feature points that define the right eye region
int RIGHT_EYE_SET[] =  {FSDKP_RIGHT_EYE, FSDKP_RIGHT_EYE_INNER_CORNER, FSDKP_RIGHT_EYE_OUTER_CORNER,
	FSDKP_RIGHT_EYE_LOWER_LINE1, FSDKP_RIGHT_EYE_LOWER_LINE2, FSDKP_RIGHT_EYE_LOWER_LINE3,
	FSDKP_RIGHT_EYE_UPPER_LINE1, FSDKP_RIGHT_EYE_UPPER_LINE2, FSDKP_RIGHT_EYE_UPPER_LINE3,
    FSDKP_RIGHT_EYE_LEFT_IRIS_CORNER, FSDKP_RIGHT_EYE_RIGHT_IRIS_CORNER};

// Returns the square of a value
inline float sqr(float v) { return v*v; }

// Enumeration of face display types for visual feedback
enum FaceType
{
    ft_normal = 0,    // Normal face detection state
    ft_active = 1,    // Face is active (mouse hover)
    ft_captured = 2   // Face is captured for liveness check
};

// Low pass filter class to stabilize frame size and smooth value changes
class LowPassFilter // low pass filter to stabilize frame size
{
    float m_a, m_y;
public:
    LowPassFilter(float a = 0.35f): m_a(a), m_y(0) {}
	float operator () (float x) {
        if(m_y==0) m_y = x;
        m_y = m_a*x + (1-m_a)*m_y;
        return m_y;
    };
};

// Initializes the FaceSDK library, camera, and tracker
// Returns true on success, false on failure
bool init_facesdk() {
    cout << "Initializing FSDK... ";
    if (FSDK_ActivateLibrary("INSERT THE LICENSE KEY HERE") != FSDKE_OK) {
        cout << "Please insert the license key in the FSDK_ActivateLibrary() function.";
        return false;
    }
    char buf[512] = { 0 };
    if (FSDK_Initialize(buf) != FSDKE_OK) {
        cout << "FaceSDK is not initialized\n";
        return false;
    }
    if (FSDK_GetLicenseInfo(buf) == FSDKE_OK)
        cout << "OK\nLicense info: " << buf << endl;
    int res = FSDK_InitializeCapturing();
    if (res != FSDKE_OK) {
        cout << "Cannot initialize capturing. Error code = " << res << endl;
        return false;
    }
    cout << "Looking for video cameras...\n";
    wchar_t** cameraList;
    int cameraCount;
    res = FSDK_GetCameraList(&cameraList, &cameraCount);
    if (res != FSDKE_OK) {
        cout << "Cannot get camera list. Error Code = " << res << endl;
        return false;
    }
    if (cameraCount == 0) {
        cout << "Please attach camera\n";
        return false;
    }
    wchar_t* cameraName = cameraList[0];
    wcout << "Using " << cameraName << " camera\n";

    FSDK_VideoFormatInfo* videoFormatList;
    int videoFormatCount;
    FSDK_GetVideoFormatList(cameraName, &videoFormatList, &videoFormatCount);

    int formatIndex = -1;
    int max_width = 0, max_height = 0;

    for (int i = 0; i < videoFormatCount; i++) {
        if (videoFormatList[i].Width > max_width && videoFormatList[i].Height > max_height) {
            max_width = videoFormatList[i].Width;
            max_height = videoFormatList[i].Height;
            formatIndex = i;
        }
        if (videoFormatList[i].BPP == 24) {
            formatIndex = i;
            break;
        }
    }

    if (formatIndex >= 0) {
        cout << "Selected video format:"
            << " Width=" << videoFormatList[formatIndex].Width
            << " Height=" << videoFormatList[formatIndex].Height
            << " BPP=" << videoFormatList[formatIndex].BPP << endl;
        cout << "Opening video camera... ";
        videoFormat = videoFormatList[formatIndex];
        FSDK_SetVideoFormat(cameraName, videoFormat);
        res = FSDK_OpenVideoCamera(cameraName, &cameraHandle);
        if (res != FSDKE_OK) {
            cout << "Cannot open the camera. Error code = " << res << endl;
            return false;
        }
        cout << "OK\nExecute the application\n";
    }

    FSDK_FreeVideoFormatList(videoFormatList);
    FSDK_FreeCameraList(cameraList, cameraCount);

    if( cameraHandle == -1) {
        cout << "Cannot find appropriate video format (RGB24)\n";
        return false;
    }

    if( FSDKE_OK != FSDK_CreateTracker(&tracker) )
    {
        cout << "Cannot create tracker\n";
        return false;
    }

    int errorPos;
    res = FSDK_SetTrackerMultipleParameters(tracker, "RecognizeFaces=true; DetectFacialFeatures=true;"
	    "HandleArbitraryRotations=true; DetermineFaceRotationAngle=true;"
	    "InternalResizeWidth=256; FaceDetectionThreshold=5;"
	    "DetectGender=true; DetectAge=true; DetectExpression=true; DetectAngles=true", &errorPos);
    if(res != FSDKE_OK) {
        cout << "Error in parameters list at position " << errorPos << endl;
        return false;
    }
    return true;
}

// Initializes the GDI+ graphics library
void init_gdiplus()
{
	Gdiplus::GdiplusStartupInput gdiplusStartupInput;
    Gdiplus::GdiplusStartup(&gdip_token, &gdiplusStartupInput, 0);
}

// Shuts down the GDI+ graphics library
void close_gdiplus()
{
    Gdiplus::GdiplusShutdown(gdip_token);
}

// Calculates the center point of a set of facial feature points
// Parameters: f - facial features, set - array of feature indices, count - number of points
// Returns: the center point as a PointF
Gdiplus::PointF dot_center(const FSDK_Features & f, const int * set, int count) 
{
    Gdiplus::PointF p = {0, 0};
    for(int i = 0; i < count; i++) {
        p.X += f[set[i]].x;
        p.Y += f[set[i]].y;
    }
    p.X /= count;
    p.Y /= count;
    return p;
}

// Class for tracking and managing individual face locations and liveness detection
class FaceLocator
{
    LowPassFilter m_w;
    Gdiplus::PointF m_center;
    float m_angle;
    Gdiplus::RectF m_frame;

    static char cmd_list[6]; // {st_L, st_R, st_F, st_U, st_D, st_S};
    string m_commands_to_do;
    int m_check_cmd;
    int m_state;
    enum {
        st_L = 1,
        st_R = 2,
        st_F = 4,
        st_U = 8,
        st_D = 16,
        st_S = 32,
    };
    const wchar_t * cmd_name(int cmd) const {
        switch(cmd) {
	        case st_L: return L"Turn left";
	        case st_R: return L"Turn right";
	        case st_F: return L"Look straight";
	        case st_U: return L"Look up";
	        case st_D: return L"Look down";
	        case st_S: return L"Make a smile";
            default: return L"Do nothing";
        }
    }

    bool on_commands() const {
        return m_check_cmd >= 0 && m_check_cmd < (int) m_commands_to_do.length();
    }

    void update_state() {
        char buf[512];
        FSDK_GetTrackerFacialAttribute(tracker, 0, m_id, "Expression", buf, sizeof(buf));
        float smile = (float) atof(strstr(buf, "Smile=")+6);
        FSDK_GetTrackerFacialAttribute(tracker, 0, m_id, "Angles", buf, sizeof(buf));
        float pan = (float) atof(strstr(buf, "Pan=")+4);
        float tilt = (float) atof(strstr(buf, "Tilt=")+5);
        cout << "Smile = " << smile << "; Pan = " << pan << "; Tilt = " << tilt << "          \r";

        int st = 0;
        if(fabs(pan) < HOR_CONF_LEVEL[0]) st = st_F;
        else if(fabs(pan) > HOR_CONF_LEVEL[1]) st = pan > 0 ? st_R : st_L;
        if(st) m_state = m_state & ~(st_L | st_R) | st;
        st = 0;
        if(tilt > UP_CONF_LEVEL[1]) st = st_U;
        else if(tilt < DOWN_CONF_LEVEL[1]) st = st_D;
        else if(tilt > DOWN_CONF_LEVEL[0] && tilt < UP_CONF_LEVEL[0]) st = st_F;
        if(st) m_state = m_state & ~(st_U | st_D) | st;

        if( m_state & (st_L | st_R | st_U | st_D) )
            m_state &= ~st_F;

        if( smile < SMILE_CONF_LEVEL[0] ) m_state &= ~st_S;
        else if( smile > SMILE_CONF_LEVEL[1] ) m_state |= st_S;
    }

    bool cmd_approved(int cmd) {
        update_state();
        if( !(cmd & m_state)) return false;
        if( cmd == st_S ) return true;
        return !(st_S & m_state);
    }

public:
    int m_id;
    int m_missed;

    FaceLocator() : m_id(-1), m_missed(0), m_check_cmd(-1) {}
    FaceLocator(int id) : m_id(id), m_missed(0), m_check_cmd(-1) {}

    void pass(HTracker tracker) 
    {
        m_missed = 0;
        FSDK_Features features;
        FSDK_GetTrackerFacialFeatures(tracker, 0, m_id, &features);
        Gdiplus::PointF cl = dot_center(features, LEFT_EYE_SET, sizeof(LEFT_EYE_SET)/sizeof(LEFT_EYE_SET[0]));
        Gdiplus::PointF cr = dot_center(features, RIGHT_EYE_SET, sizeof(RIGHT_EYE_SET)/sizeof(RIGHT_EYE_SET[0]));

        float w = m_w(sqrtf(sqr(cr.X - cl.X) + sqr(cr.Y - cl.Y))*2.6f);
        float h = w*1.4f;

        m_center = {(cr.X + cl.X)/2, (cr.Y + cl.Y)/2 + w*0.05f};
        m_angle = atan2(cr.Y-cl.Y, cr.X-cl.X)*180/PI;
        m_frame = {-w/2, -h/2, w, h};

        if( on_commands() && cmd_approved(m_commands_to_do[m_check_cmd]) )
            m_check_cmd++;
    }

    bool is_inside(int x, int y) const {
        float a = m_angle*PI/180;
        float xx = (x-m_center.X)*cosf(a) + (y-m_center.Y)*sinf(a);
        float yy = (x-m_center.X)*sinf(a) - (y-m_center.Y)*cosf(a);
        return sqr(xx/m_frame.X) + sqr(yy/m_frame.Y) <= 1;
    }

    void draw(Gdiplus::Graphics * surf, FaceType ft)
    {
        Gdiplus::GraphicsContainer gc = surf->BeginContainer();
        surf->TranslateTransform(m_center.X, m_center.Y);
        surf->RotateTransform(m_angle);
        Gdiplus::Pen pen(0x60ffffff, 5);
        surf->DrawEllipse(&pen, m_frame);
        const wchar_t * name = 0;
        wchar_t buf[256];
        if(ft==ft_active) {
            pen.SetColor(0xFF00ff00);
            pen.SetWidth(1);
            surf->DrawEllipse(&pen, m_frame);
            name = L"Press to check liveness";
        }
        if(ft==ft_captured) {
            if(m_check_cmd == m_commands_to_do.length()) {
                pen.SetColor(0xFF00ff00);
                pen.SetWidth(4);
                name = L"LIVE";
            }
            else {
                pen.SetColor(0xFFff0000);
                pen.SetWidth(2);
                swprintf_s(buf, L"%s (%d of %d)", cmd_name(m_commands_to_do[m_check_cmd]), m_check_cmd+1, (int) m_commands_to_do.length());
                name = buf;
            }
            surf->DrawEllipse(&pen, m_frame);
        }
        surf->EndContainer(gc);
		if(name) {
            Gdiplus::SolidBrush text_color(0xff808080);
            Gdiplus::PointF o(m_center.X+m_frame.X+2, m_center.Y+m_frame.Y+2);
			surf->DrawString(name, (int) wcslen(name), font, o, &text_color); 
            o.X -= 2; o.Y -=2;
            text_color.SetColor(0xffffffff);
			surf->DrawString(name, (int) wcslen(name), font, o, &text_color); 
        }
    }

    void check_liveness(bool f)
    {
        if(f) {
            m_state = st_F;
            m_commands_to_do = st_F; // first command is always 'look forward'
            while(m_commands_to_do.length() < NUM_OF_COMMANDS) {
                char cmd = cmd_list[std::rand()*sizeof(cmd_list)/(RAND_MAX+1)];
                if(m_commands_to_do.back() != cmd)
                    m_commands_to_do.push_back(cmd);
            }
            if( m_commands_to_do.find_first_of(st_S) == string::npos ) // make sure the 'smile' command is the list
                m_commands_to_do[ 1+std::rand()*(NUM_OF_COMMANDS-1)/(RAND_MAX+1) ] = st_S;
            m_check_cmd = 0;
        } else {
            m_commands_to_do.clear();
            m_check_cmd = -1;
        }
    }
};

// Static list of available liveness check commands
char FaceLocator::cmd_list[6] = {st_L, st_R, st_F, st_U, st_D, st_S};

// Class for managing multiple face locators and user interaction
class FaceLocators
{
    unordered_map<int, FaceLocator> m_locator;
    int m_activeFace;
    int m_capturedFace;
    POINT m_mouse;
public:
    FaceLocators() : m_capturedFace(-1), m_activeFace(-1) {}

    void capture_face(int id)
    {
        if(m_capturedFace >= 0) {
            unordered_map<int, FaceLocator>::iterator it = m_locator.find(m_capturedFace);
            if( it != m_locator.end() ) it->second.check_liveness(false);
        }
        m_capturedFace = id;
        if(m_capturedFace >= 0) {
            unordered_map<int, FaceLocator>::iterator it = m_locator.find(id);
            if( it != m_locator.end() ) it->second.check_liveness(true);
        }
    }

    void addFaces(long long * Ids, int count, Gdiplus::Graphics * surf) {
        for(unordered_map<int, FaceLocator>::iterator it = m_locator.begin(); it != m_locator.end(); it++)
            it->second.m_missed++;
        for(int i = 0; i < count; i++) {
            unordered_map<int, FaceLocator>::iterator it = m_locator.find((int) Ids[i]);
		    if( it == m_locator.end() )
                m_locator[(int) Ids[i]] = FaceLocator((int) Ids[i]);
            m_locator[(int) Ids[i]].pass(tracker);
        }
        for(unordered_map<int, FaceLocator>::iterator it = m_locator.begin(); it != m_locator.end(); )
            if( it->second.m_missed > FRAMES_TO_KEEP_MISSED_FACE) {
                if(it->first == m_capturedFace ) capture_face(-1);
                it = m_locator.erase(it);
            }
            else it++;
    }
    void set_mouse(POINT mouse) { m_mouse = mouse; }
    void draw(Gdiplus::Graphics * surf) 
    {
        m_activeFace = -1;
        for(unordered_map<int, FaceLocator>::iterator it = m_locator.begin(); it != m_locator.end(); it++)
        {
            bool inside = it->second.is_inside(m_mouse.x, m_mouse.y);
            if(inside) m_activeFace = it->first;
            FaceType ft = inside ? ft_active : ft_normal;
            if( m_capturedFace == it->first) 
                ft = ft_captured;
            it->second.draw(surf, ft);
        }
    }
    void on_click() {
	    if( m_activeFace && m_capturedFace != m_activeFace)
	        capture_face(m_activeFace);
	    else
	        capture_face(-1);
    }
};

// Global instance of FaceLocators for managing all detected faces
FaceLocators locators;

// Window message processing callback function
LRESULT CALLBACK WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	if( message == WM_DESTROY) {
		need_to_exit = true;
    }
	else
		if(message == WM_LBUTTONDOWN) {
            locators.on_click();
			return 1;
        }
	return DefWindowProc(hWnd, message, wParam, lParam);
}

// Creates and initializes the main application window
// Parameters: width - window width, height - window height
// Returns: handle to the created window
HWND init_window(int width, int height)
{
	WNDCLASSEX wcex;
	wcex.cbSize = sizeof(WNDCLASSEX);
	wcex.style			= 0;
	wcex.lpfnWndProc	= WndProc;
	wcex.cbClsExtra		= 0;
	wcex.cbWndExtra		= 0;
	wcex.hInstance		= 0;
	wcex.hIcon			= 0;
	wcex.hCursor		= LoadCursor(NULL, IDC_ARROW);
	wcex.hbrBackground	= 0;
	wcex.lpszMenuName	= 0;
	wcex.lpszClassName	= L"My Window Class";
	wcex.hIconSm		= 0;
	RegisterClassEx(&wcex);

    int x = 100, y = 100;
	return CreateWindowEx(WS_EX_APPWINDOW, L"My Window Class", L"Liveness Recognition", WS_SYSMENU | WS_CAPTION | WS_CLIPCHILDREN, 
        x, y, width, height, 0, 0, 0, 0);
}

// Main application entry point
// Initializes FaceSDK, creates the window, and runs the main processing loop
int main()
{
    if( !init_facesdk() ) 
        return -1;
    hWnd = init_window(videoFormat.Width, videoFormat.Height);
    ShowWindow(hWnd, SW_SHOW);
    init_gdiplus();
    Gdiplus::Graphics * graphics = Gdiplus::Graphics::FromHWND(hWnd);
    Gdiplus::Bitmap * backsurf = new Gdiplus::Bitmap(videoFormat.Width, videoFormat.Height, graphics);
    Gdiplus::Graphics * surfGr = new Gdiplus::Graphics(backsurf);
    surfGr->SetSmoothingMode(Gdiplus::SmoothingMode::SmoothingModeHighQuality);

    font = new Gdiplus::Font(new Gdiplus::FontFamily(L"Tahoma"), FONT_SIZE);
    std::srand((int) time(0));
    
    while(1) {
        HImage image;
        HBITMAP hBmp;
        FSDK_GrabFrame(cameraHandle, &image);
        FSDK_SaveImageToHBitmap(image, &hBmp);
        Gdiplus::Bitmap * bmp = Gdiplus::Bitmap::FromHBITMAP(hBmp, 0);
        DeleteObject(hBmp);
        surfGr->DrawImage(bmp, 0, 0);
        delete bmp;

        long long IDs[256];
        long long faceCount;
        FSDK_FeedFrame(tracker, 0, image, &faceCount, IDs, sizeof(IDs));
        locators.addFaces(IDs, (int) faceCount, surfGr);
        POINT p;
        GetCursorPos(&p);
        ScreenToClient(hWnd, &p);
        locators.set_mouse(p);
        locators.draw(surfGr);

        graphics->DrawImage(backsurf, 0, 0);

        FSDK_FreeImage(image);

	    MSG msg;
	    if(PeekMessage(&msg, 0, 0, 0, PM_REMOVE)) {
		    TranslateMessage(&msg);
		    DispatchMessage(&msg);
		    if(msg.message == WM_KEYDOWN && msg.wParam == VK_ESCAPE || need_to_exit) 
                break;
        }
    }

    delete graphics;
    close_gdiplus();
}
