package com.luxand.livenessrecognition;

import com.luxand.FSDK;
import com.luxand.FSDK.HTracker;

import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.ArrayList;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import android.Manifest;
import android.app.Activity;
import android.content.Context;
import android.content.DialogInterface;
import android.content.pm.ActivityInfo;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Paint.Align;
import android.graphics.drawable.ColorDrawable;
import android.hardware.Camera;
import android.hardware.Camera.PreviewCallback;
import android.hardware.Camera.Size;
import android.os.Bundle;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.Surface;
import android.view.SurfaceHolder;
import android.view.SurfaceView;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.ViewGroup;
import android.view.Window;
import android.view.WindowManager;
import android.view.ViewGroup.LayoutParams;
import android.widget.EditText;
import android.widget.FrameLayout;
import android.widget.LinearLayout;

import android.support.v4.app.ActivityCompat;
import android.support.v4.content.ContextCompat;
import android.support.v7.app.AlertDialog;
import android.annotation.SuppressLint;
import android.text.TextPaint;
import com.luxand.livenessrecognition.R;

public class MainActivity extends Activity implements OnClickListener {

    public static final int CAMERA_PERMISSION_REQUEST_CODE = 1;

    private boolean mIsFailed = false;
    private FrameLayout mLayout;
    private Preview mPreview;
    private ProcessImageAndDrawResults mDraw;
    private final String database = "memory70.dat";

    private boolean wasStopped = false;

    public static float sDensity = 1.0f;
    
    public void showErrorAndClose(String error, int code) {
        AlertDialog.Builder builder = new AlertDialog.Builder(this);
        builder.setMessage(error + ": " + code)
            .setPositiveButton("Ok", new DialogInterface.OnClickListener() {
                @Override
                public void onClick(DialogInterface dialogInterface, int i) {
                    android.os.Process.killProcess(android.os.Process.myPid());
                }
            })
            .show();        
    }
    
    public void showMessage(String message) {
        AlertDialog.Builder builder = new AlertDialog.Builder(this);
        builder.setMessage(message)
            .setPositiveButton("Ok", new DialogInterface.OnClickListener() {
                @Override
                public void onClick(DialogInterface dialogInterface, int i) {
                }
            })
            .setCancelable(false) // cancel with button only
            .show();        
    }
    
    private void resetTrackerParameters() {
        int[] err_pos = new int[1];
        FSDK.SetTrackerMultipleParameters(mDraw.mTracker,
                "RecognizeFaces=true;" +
                        "DetectFacialFeatures=true;" +
                        "HandleArbitraryRotations=true;" +
                        "DetermineFaceRotationAngle=true;" +
                        "InternalResizeWidth=64;" +
                        "FaceDetectionThreshold=5;" +
                        "DetectGender=true;" +
                        "DetectAge=true;" +
                        "DetectExpression=true;" +
                        "DetectAngles=true;", err_pos);
        if (err_pos[0] != 0) {
            showErrorAndClose("Error setting tracker parameters, position", err_pos[0]);
        }
    }
    
    /** Called when the activity is first created. */
    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        sDensity = getResources().getDisplayMetrics().scaledDensity;
        
        int res = FSDK.ActivateLibrary("INSERT THE LICENSE KEY HERE");
        if (res != FSDK.FSDKE_OK) {
            mIsFailed = true;
            showErrorAndClose("FaceSDK activation failed", res);
        } else {
            FSDK.Initialize();
            
            // Hide the window title (it is done in manifest too)
            getWindow().setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN, WindowManager.LayoutParams.FLAG_FULLSCREEN);
            requestWindowFeature(Window.FEATURE_NO_TITLE);
            
            // Lock orientation
            setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_PORTRAIT);

            mLayout = new FrameLayout(this);
            LayoutParams params = new LayoutParams
                    (LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT);
            mLayout.setLayoutParams(params);
            setContentView(mLayout);

            checkCameraPermissionsAndOpenCamera();
        }                
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, String permissions[], int[] grantResults) {
        switch (requestCode) {
            case CAMERA_PERMISSION_REQUEST_CODE:
                openCamera();
                break;
            default:
                break;
        }
    }

    private void checkCameraPermissionsAndOpenCamera() {
        if (PackageManager.PERMISSION_GRANTED != ContextCompat.checkSelfPermission(this, Manifest.permission.CAMERA)) {
            if (ActivityCompat.shouldShowRequestPermissionRationale(this, Manifest.permission.CAMERA)) {

                final Runnable onCloseAlert = new Runnable() {
                    @Override
                    public void run() {
                        ActivityCompat.requestPermissions(MainActivity.this,
                                new String[] {Manifest.permission.CAMERA},
                                CAMERA_PERMISSION_REQUEST_CODE);
                    }
                };

                alert(this, onCloseAlert, "The application processes frames from camera.");
            } else {
                ActivityCompat.requestPermissions(this,
                        new String[] {Manifest.permission.CAMERA},
                        CAMERA_PERMISSION_REQUEST_CODE);
            }
        } else {
            openCamera();
        }
    }

    public static void alert(final Context context, final Runnable callback, String message) {
        AlertDialog.Builder dialog = new AlertDialog.Builder(context);
        dialog.setMessage(message);
        dialog.setNegativeButton("Ok", new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                dialog.dismiss();
            }
        });
        if (callback != null) {
            dialog.setOnDismissListener(new DialogInterface.OnDismissListener() {
                @Override
                public void onDismiss(DialogInterface dialog) {
                    callback.run();
                }
            });
        }
        dialog.show();
    }

    private void openCamera() {
        // Camera layer and drawing layer
        View background = new View(this);
        background.setBackgroundColor(Color.BLACK);
        mDraw = new ProcessImageAndDrawResults(this);
        mPreview = new Preview(this, mDraw);
        //mPreview.setBackgroundColor(Color.GREEN);
        //mDraw.setBackgroundColor(Color.RED);
        mDraw.mTracker = new HTracker();
        String templatePath = this.getApplicationInfo().dataDir + "/" + database;
        if (FSDK.FSDKE_OK != FSDK.LoadTrackerMemoryFromFile(mDraw.mTracker, templatePath)) {
            int res = FSDK.CreateTracker(mDraw.mTracker);
            if (FSDK.FSDKE_OK != res) {
                showErrorAndClose("Error creating tracker", res);
            }
        }

        resetTrackerParameters();

        this.getWindow().setBackgroundDrawable(new ColorDrawable()); //black background

        mLayout.setVisibility(View.VISIBLE);
        addContentView(background, new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT));
        addContentView(mPreview, new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT)); //creates MainActivity contents
        addContentView(mDraw, new LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT));

        // Menu
        LayoutInflater inflater = (LayoutInflater)this.getSystemService( Context.LAYOUT_INFLATER_SERVICE );
        View buttons = inflater.inflate(R.layout.bottom_menu, null );
        buttons.findViewById(R.id.helpButton).setOnClickListener(this);
        buttons.findViewById(R.id.clearButton).setOnClickListener(this);
        addContentView(buttons, new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT));
    }

    @Override
    public void onClick(View view) {
        if (view.getId() == R.id.helpButton) {
            showMessage("Luxand Liveness Recognition\n\nJust tap any detected face to check liveness. " +
                        "The app will prompt you with certain commands to check for liveness." +
                        "For best results, hold the device at arm's length. " +
                        "The SDK is available for mobile developers: www.luxand.com/facesdk"
            );
        } else if (view.getId() == R.id.clearButton) {
            AlertDialog.Builder builder = new AlertDialog.Builder(this);
            builder.setMessage("Are you sure to clear the memory?" )
                .setPositiveButton("Ok", new DialogInterface.OnClickListener() {
                    @Override public void onClick(DialogInterface dialogInterface, int j) {
                        pauseProcessingFrames();
                        FSDK.ClearTracker(mDraw.mTracker);
                        resetTrackerParameters();
                        resumeProcessingFrames();
                    }
                })
                .setNegativeButton("Cancel", new DialogInterface.OnClickListener() {
                    @Override public void onClick(DialogInterface dialogInterface, int j) {
                    }
                })
                .setCancelable(false) // cancel with button only
                .show();
        }
    }

    @Override
    protected void onStop() {
        if (mDraw != null || mPreview != null) {
            mPreview.setVisibility(View.GONE); // to destroy surface
            mLayout.setVisibility(View.GONE);
            mLayout.removeAllViews();
            mPreview.releaseCallbacks();
            mPreview = null;
            mDraw = null;
            wasStopped = true;
        }
        super.onStop();
    }

    @Override
    protected void onStart() {
        super.onStart();
        if (wasStopped && mDraw == null) {
            checkCameraPermissionsAndOpenCamera();
            //openCamera();
            wasStopped = false;
        }
    }

    @Override
    public void onPause() {
        super.onPause();
        if (mDraw != null) {
            pauseProcessingFrames();
            String templatePath = this.getApplicationInfo().dataDir + "/" + database;
            FSDK.SaveTrackerMemoryToFile(mDraw.mTracker, templatePath);
        }
    }
    
    @Override
    public void onResume() {
        super.onResume();
        if (mIsFailed)
            return;
        resumeProcessingFrames();
    }
    
    private void pauseProcessingFrames() {
        if (mDraw != null) {
            mDraw.mStopping = 1;

            // It is essential to limit wait time, because mStopped will not be set to 0, if no frames are feeded to mDraw
            for (int i = 0; i < 100; ++i) {
                if (mDraw.mStopped != 0) break;
                try {
                    Thread.sleep(10);
                } catch (Exception ignored) {
                }
            }
        }
    }
    
    private void resumeProcessingFrames() {
        if (mDraw != null) {
            mDraw.mStopped = 0;
            mDraw.mStopping = 0;
        }
    }
}



class FaceRectangle {

    public float x1, y1, x2, y2;

    public void set(float w, float h) {
        x1 = -w / 2;
        y1 = -h / 2;
        x2 =  w / 2;
        y2 =  h / 2;
    }
}

class LowPassFilter {

    private float a;
    private Float y = null;

    public LowPassFilter() {
        a = 0.325f;
    }

    public float pass(final float x) {
        y = a * x + (1 - a) * (y == null ? x : y);
        return y;
    }

}

class Point {

    public float x;
    public float y;

    public Point() {
        x = 0;
        y = 0;
    }

}

class FaceLocator {

    private enum Command {
        TURN_LEFT,
        TURN_RIGHT,
        LOOK_STRAIGHT,
        LOOK_UP,
        LOOK_DOWN,
        SMILE
    }

    private static Set<Command> LEFT_RIGHT = new HashSet<Command>() {{ add(Command.TURN_LEFT); add(Command.TURN_RIGHT); }};
    private static Set<Command> UP_DOWN = new HashSet<Command>() {{ add(Command.LOOK_DOWN); add(Command.LOOK_UP); }};

    private static int[] LEFT_EYE = new int[] {
            FSDK.FSDKP_LEFT_EYE,
            FSDK.FSDKP_LEFT_EYE_INNER_CORNER,
            FSDK.FSDKP_LEFT_EYE_OUTER_CORNER,
            FSDK.FSDKP_LEFT_EYE_LOWER_LINE1,
            FSDK.FSDKP_LEFT_EYE_LOWER_LINE2,
            FSDK.FSDKP_LEFT_EYE_LOWER_LINE3,
            FSDK.FSDKP_LEFT_EYE_UPPER_LINE1,
            FSDK.FSDKP_LEFT_EYE_UPPER_LINE2,
            FSDK.FSDKP_LEFT_EYE_UPPER_LINE3,
            FSDK.FSDKP_LEFT_EYE_RIGHT_IRIS_CORNER,
            FSDK.FSDKP_LEFT_EYE_LEFT_IRIS_CORNER
    };

    private static int[] RIGHT_EYE = new int[] {
            FSDK.FSDKP_RIGHT_EYE,
            FSDK.FSDKP_RIGHT_EYE_INNER_CORNER,
            FSDK.FSDKP_RIGHT_EYE_OUTER_CORNER,
            FSDK.FSDKP_RIGHT_EYE_LOWER_LINE1,
            FSDK.FSDKP_RIGHT_EYE_LOWER_LINE2,
            FSDK.FSDKP_RIGHT_EYE_LOWER_LINE3,
            FSDK.FSDKP_RIGHT_EYE_UPPER_LINE1,
            FSDK.FSDKP_RIGHT_EYE_UPPER_LINE2,
            FSDK.FSDKP_RIGHT_EYE_UPPER_LINE3,
            FSDK.FSDKP_RIGHT_EYE_RIGHT_IRIS_CORNER,
            FSDK.FSDKP_RIGHT_EYE_LEFT_IRIS_CORNER
    };

    private static Random random = new Random();

    private static Paint facePaint = new Paint();
    private static Paint activePaint = new Paint();
    private static Paint featuresPaint = new Paint();
    private static TextPaint textPaint = new TextPaint();
    private static TextPaint greenPaint = new TextPaint();
    private static TextPaint shadowPaint = new TextPaint();

    static {
    	facePaint.setColor(Color.WHITE);
    	facePaint.setStyle(Paint.Style.STROKE);
    	facePaint.setStrokeWidth(7);

    	activePaint.setColor(Color.RED);
    	activePaint.setStyle(Paint.Style.STROKE);
    	activePaint.setStrokeWidth(4);

    	featuresPaint.setColor(Color.YELLOW);
    	featuresPaint.setStyle(Paint.Style.FILL);

    	textPaint.setColor(Color.WHITE);
    	textPaint.setStyle(Paint.Style.FILL);
    	textPaint.setTextSize(18 * MainActivity.sDensity);

    	greenPaint.setColor(Color.GREEN);
    	greenPaint.setStyle(Paint.Style.FILL);
    	greenPaint.setTextSize(18 * MainActivity.sDensity);

        shadowPaint.setColor(Color.BLACK);
        shadowPaint.setStyle(Paint.Style.FILL);
        shadowPaint.setTextSize(18 * MainActivity.sDensity);
	}

    private long faceId;
    private int commandId;
    private int countdown = 35;

    private float ratio;
    private double angle;

    private FaceRectangle frame = new FaceRectangle();

    private boolean live = false;

    private Point center = new Point();

    private LowPassFilter lpf = null;
    private LowPassFilter pan = new LowPassFilter();
    private LowPassFilter tilt = new LowPassFilter();

    private HTracker tracker;

    private List<Command> commands;
    private Set<Command> state = new HashSet<Command>() {{ add(Command.LOOK_STRAIGHT); }};

    private boolean active   = false;

    private static Map<Command, String> commandNames = new HashMap<Command, String>() {{
        put(Command.TURN_LEFT, "Turn Left");
        put(Command.TURN_RIGHT, "Turn Right");
        put(Command.LOOK_STRAIGHT, "Look straight");
        put(Command.LOOK_UP, "Look up");
        put(Command.LOOK_DOWN, "Look down");
        put(Command.SMILE, "Make a smile");
    }};

    public FaceLocator(long faceId, final HTracker tracker, final float ratio) {
        this.faceId = faceId;
        this.tracker = tracker;
        this.ratio = ratio;
    }

    public boolean doesIntersect(final FaceLocator other) {
        return !(
                frame.x1 >= other.frame.x2 ||
                        frame.x2 < other.frame.x1 ||
                        frame.y1 >= other.frame.y2 ||
                        frame.y2 < other.frame.y1
        );
    }

    public boolean isInside(double x, double y) {
        x -= center.x * ratio;
        y -= center.y * ratio;

        final double a = angle * Math.PI / 180;
        final double xx = x * Math.cos(a) + y * Math.sin(a);
        final double yy = x * Math.sin(a) - y * Math.cos(a);

        return Math.pow(xx / frame.x1 / ratio, 2) + Math.pow(yy / frame.y1 / ratio, 2) <= 1;
    }

    public boolean isActive() {
        return active;
    }

    public void setActive(boolean active) {
        this.active = active;
        if (!active)
            return;

        live = false;
        commandId = 0;

        commands = new ArrayList<Command>() {{ add(Command.LOOK_STRAIGHT); }};

        while (commands.size() < 6) {
            Command cmd = Command.values()[random.nextInt(6)];
            if (commands.get(commands.size() - 1) != cmd)
                commands.add(cmd);
        }

        if (!commands.contains(Command.SMILE))
            commands.set(random.nextInt(commands.size() - 1), Command.SMILE);
    }

    private boolean approveCommand(final Command command) {
        if (!state.contains(command))
            return false;

        if (command == Command.SMILE)
            return true;

        return !state.contains(Command.SMILE);
    }

    public void updateState() {
        final String[] result = new String[] { "" };
        if (FSDK.GetTrackerFacialAttribute(tracker, 0, faceId, "Expression", result, 256) != FSDK.FSDKE_OK)
            return;

        double smile = 0;

        for (String pair : result[0].split(";")) {
            String[] values = pair.split("=");
            if (values[0].equalsIgnoreCase("smile")) {
                smile = Double.parseDouble(values[1]);
                break;
            }
        }

        double pan = 0;
        double tilt = 0;

        if (FSDK.GetTrackerFacialAttribute(tracker, 0, faceId, "Angles", result, 256) != FSDK.FSDKE_OK)
            return;

        for (String pair : result[0].split(";")) {
            String[] values = pair.split("=");

            if (values[0].equalsIgnoreCase("pan")) {
                pan = this.pan.pass(Float.parseFloat(values[1]));
                continue;
            }

            if (values[0].equalsIgnoreCase("tilt")) {
                tilt = this.tilt.pass(Float.parseFloat(values[1]));
            }
        }

        Command st = null;

        final double HOR_CONF_LEVEL_LOW = 2;
        final double HOR_CONF_LEVEL_HIGH = 17;

        if (Math.abs(pan) < HOR_CONF_LEVEL_LOW)
            st = Command.LOOK_STRAIGHT;
        else if (Math.abs(pan) > HOR_CONF_LEVEL_HIGH)
            st = pan > 0 ? Command.TURN_LEFT : Command.TURN_RIGHT; // swapped for mirrored feed from front camera

        if (st != null) {
            state.removeAll(LEFT_RIGHT);
            state.add(st);
        }

        final double UP_CONF_LEVEL_LOW = 2;
        final double UP_CONF_LEVEL_HIGH = 5;
        final double DOWN_CONF_LEVEL_LOW = -2;
        final double DOWN_CONF_LEVEL_HIGH = -5;

        st = null;

        if (tilt > UP_CONF_LEVEL_HIGH)
            st = Command.LOOK_UP;
        else if (tilt < DOWN_CONF_LEVEL_HIGH)
            st = Command.LOOK_DOWN;
        else if (tilt > DOWN_CONF_LEVEL_LOW && tilt < UP_CONF_LEVEL_LOW)
            st = Command.LOOK_STRAIGHT;

        if (st != null) {
            state.removeAll(UP_DOWN);
            state.add(st);
        }

        if (state.contains(Command.LOOK_UP)   || state.contains(Command.LOOK_DOWN) ||
                state.contains(Command.TURN_LEFT) || state.contains(Command.TURN_RIGHT))
            state.remove(Command.LOOK_STRAIGHT);

        final double SMILE_CONF_LEVEL_LOW = 0.3;
        final double SMILE_CONF_LEVEL_HIGH = 0.6;

        if (smile < SMILE_CONF_LEVEL_LOW)
            state.remove(Command.SMILE);
        else if (smile > SMILE_CONF_LEVEL_HIGH)
            state.add(Command.SMILE);

        if (active && !live)
            if (approveCommand(commands.get(commandId))) {
                ++commandId;
                if (commandId >= commands.size())
                    live = true;
            }
    }

    private static FSDK.TPoint dotCenter(FSDK.TPoint[] points) {
        FSDK.TPoint result = new FSDK.TPoint() {{
            x = 0;
            y = 0;
        }};


        for (FSDK.TPoint point : points) {
            result.x += point.x;
            result.y += point.y;
        }

        result.x /= points.length;
        result.y /= points.length;

        return result;
    }

    private static FSDK.TPoint[] getPoints(FSDK.FSDK_Features features, int[] indices) {
        FSDK.TPoint[] result = new FSDK.TPoint[indices.length];

        for (int i = 0; i < result.length; ++i)
            result[i] = features.features[indices[i]];

        return result;
    }

    private void drawShape(Canvas canvas) {
    	float left   = (center.x + frame.x1) * ratio;
		float top    = (center.y + frame.y1) * ratio;
		float right  = (center.x + frame.x2) * ratio;
		float bottom = (center.y + frame.y2) * ratio;

        canvas.drawOval(left, top, right, bottom, facePaint);

        if (active)
            canvas.drawOval(left, top, right, bottom, activePaint);
    }

    public boolean draw(Canvas canvas) {
        FSDK.FSDK_Features features = new FSDK.FSDK_Features();

        if (FSDK.GetTrackerFacialFeatures(tracker, 0, faceId, features) != FSDK.FSDKE_OK) {
            countdown -= 1;

            if (countdown <= 8) {
                frame.x1 *= 0.95;
                frame.y1 *= 0.95;
                frame.x2 *= 0.95;
                frame.y2 *= 0.95;
            }

            drawShape(canvas);
            return countdown > 0;
        }

        if (lpf == null)
            lpf = new LowPassFilter();

        FSDK.TPoint left = dotCenter(getPoints(features, LEFT_EYE));
        FSDK.TPoint right = dotCenter(getPoints(features, RIGHT_EYE));

        float w = lpf.pass((right.x - left.x) * 2.8f);
        float h = w * 1.4f;

        center.x = (right.x + left.x) / 2.f;
        center.y = (right.y + left.y) / 2.f + w * 0.05f;
        angle = Math.atan2(right.y - left.y, right.x - left.x) * 180 / Math.PI;

        frame.set(w, h);

        drawShape(canvas);

        if (!active)
            return true;

// Draw facial features
//        for (FSDK.TPoint point : features.features)
//            canvas.drawCircle(point.x * ratio, point.y * ratio, 2, featuresPaint);

        String name = live ? "LIVE!" : "";
        TextPaint style = live ? greenPaint : textPaint;

        if (isActive())
            if (commandId >= commands.size())
                name = "LIVE!";
            else
                name = String.format("%s (%s of %s)",
                        commandNames.get(commands.get(commandId)), commandId + 1, commands.size());

        if (name.length() > 0) {
            canvas.drawText(name, center.x - w / 2 + 5, center.y - h / 2 + 5, shadowPaint);
            canvas.drawText(name, center.x - w / 2, center.y - h / 2, style);
        }

        updateState();
        countdown = 35;

        return true;
    }


}

// Draw graphics on top of the video
class ProcessImageAndDrawResults extends View {
    public HTracker mTracker;
    
    final int MAX_FACES = 5;
    int mStopping;
    int mStopped;
    
    Context mContext;
    byte[] mYUVData;
    byte[] mRGBData;
    int mImageWidth, mImageHeight;
    boolean first_frame_saved;
    boolean rotated;

    final Lock faceLock = new ReentrantLock();

    FSDK.HImage image = new FSDK.HImage();
    FSDK.HImage rotatedImage = new FSDK.HImage();

    FSDK.FSDK_IMAGEMODE image_mode = new FSDK.FSDK_IMAGEMODE() {{
        mode = FSDK.FSDK_IMAGEMODE.FSDK_IMAGE_COLOR_24BIT;
    }};

    final long[] IDs = new long[MAX_FACES];
    final long[] face_count = new long[1];
    final ArrayList<Long> missed = new ArrayList<>();

    final Map<Long, FaceLocator> trackers = new HashMap<>();


    public ProcessImageAndDrawResults(Context context) {
        super(context);
        
        mStopping = 0;
        mStopped = 0;
        rotated = false;
        mContext = context;
        
        //mBitmap = null;
        mYUVData = null;
        mRGBData = null;
        
        first_frame_saved = false;
    }

    private static boolean contains(long[] ids, long id) {
        for (long di : ids)
            if (id == di)
                return true;
        return false;
    }

    @SuppressLint("DrawAllocation")
    @Override
    protected void onDraw(Canvas canvas) {
        if (mStopping == 1) {
            mStopped = 1;
            super.onDraw(canvas);
            return;
        }
        
        if (mYUVData == null) {
            super.onDraw(canvas);
            return; //nothing to process
        }
        
        int canvasWidth = getWidth();
        //int canvasHeight = canvas.getHeight();
        
        // Convert from YUV to RGB
        decodeYUV420SP(mRGBData, mYUVData, mImageWidth, mImageHeight);
        
        // Load image to FaceSDK
        FSDK.LoadImageFromBuffer(image, mRGBData, mImageWidth, mImageHeight, mImageWidth * 3, image_mode);
        FSDK.MirrorImage(image, false);
        
        //it is necessary to work with local variables (onDraw called not the time when mImageWidth,... being reassigned, so swapping mImageWidth and mImageHeight may be not safe)
        int ImageWidth = mImageWidth;
        //int ImageHeight = mImageHeight;
        if (rotated) {
            //noinspection SuspiciousNameCombination
            ImageWidth = mImageHeight;
            //ImageHeight = mImageWidth;

            FSDK.CreateEmptyImage(rotatedImage);
            FSDK.RotateImage90(image, -1, rotatedImage);
            FSDK.FreeImage(image);

            FSDK.HImage tmp = image;
            image = rotatedImage;
            rotatedImage = tmp;
        }

        // Save first frame to gallery to debug (e.g. rotation angle)
        /*
        if (!first_frame_saved) {                
            first_frame_saved = true;
            String galleryPath = Environment.getExternalStoragePublicDirectory(Environment.DIRECTORY_PICTURES).getAbsolutePath();
            FSDK.SaveImageToFile(RotatedImage, galleryPath + "/first_frame.jpg"); //frame is rotated!
        }
        */

        Arrays.fill(IDs, 0);
        FSDK.FeedFrame(mTracker, 0, image, face_count, IDs);
        FSDK.FreeImage(image);

        float ratio = (canvasWidth * 1.0f) / ImageWidth;

        faceLock.lock();

        // Mark and name faces
        for (int i = 0; i < face_count[0]; ++i)
            if (!trackers.containsKey(IDs[i]))
                trackers.put(IDs[i], new FaceLocator(IDs[i], mTracker, ratio));

        missed.clear();
        for (Map.Entry<Long, FaceLocator> entry : trackers.entrySet())
            if (contains(IDs, entry.getKey()))
                entry.getValue().draw(canvas);
            else
                missed.add(entry.getKey());

        for (long id : missed) {
            FaceLocator st = trackers.get(id);

            for (int i = 0; i < face_count[0]; ++i)
                if (IDs[i] != id && st.doesIntersect(trackers.get(IDs[i]))) {
                    trackers.remove(id);
                    break;
                }

            if (trackers.containsKey(id) && !st.draw(canvas))
                trackers.remove(id);
        }

        faceLock.unlock();
        
        super.onDraw(canvas);      
    } // end onDraw method

    
    @SuppressLint("ClickableViewAccessibility")
    @Override
    public boolean onTouchEvent(MotionEvent event) { //NOTE: the method can be implemented in Preview class
        if (event.getAction() == MotionEvent.ACTION_DOWN) {
            int x = (int) event.getX();
            int y = (int) event.getY();

            for (FaceLocator loc : trackers.values())
                if (loc.isInside(x, y)) {
                    if (!loc.isActive())
                        loc.setActive(true);
                } else
                    loc.setActive(false);
        }
        return true;
    }
    
    static public void decodeYUV420SP(byte[] rgb, byte[] yuv420sp, int width, int height) {
        final int frameSize = width * height;
        int yp = 0;
        for (int j = 0; j < height; j++) {
            int uvp = frameSize + (j >> 1) * width, u = 0, v = 0;
            for (int i = 0; i < width; i++) {
                int y = (0xff & ((int) yuv420sp[yp])) - 16;
                if (y < 0) y = 0;
                if ((i & 1) == 0) {
                    v = (0xff & yuv420sp[uvp++]) - 128;
                    u = (0xff & yuv420sp[uvp++]) - 128;
                }    
                int y1192 = 1192 * y;
                int r = (y1192 + 1634 * v);
                int g = (y1192 - 833 * v - 400 * u);
                int b = (y1192 + 2066 * u);
                if (r < 0) r = 0; else if (r > 262143) r = 262143;
                if (g < 0) g = 0; else if (g > 262143) g = 262143;
                if (b < 0) b = 0; else if (b > 262143) b = 262143;
                
                rgb[3*yp] = (byte) ((r >> 10) & 0xff);
                rgb[3*yp+1] = (byte) ((g >> 10) & 0xff);
                rgb[3*yp+2] = (byte) ((b >> 10) & 0xff);
                ++yp;
            }
        }
    }  
} // end of ProcessImageAndDrawResults class




// Show video from camera and pass frames to ProcessImageAndDraw class 
class Preview extends SurfaceView implements SurfaceHolder.Callback {
    Context mContext;
    SurfaceHolder mHolder;
    Camera mCamera;
    ProcessImageAndDrawResults mDraw;
    boolean mFinished;
    boolean mIsCameraOpen = false;

    boolean mIsPreviewStarted = false;

    Preview(Context context, ProcessImageAndDrawResults draw) {
        super(context);      
        mContext = context;
        mDraw = draw;
        
        //Install a SurfaceHolder.Callback so we get notified when the underlying surface is created and destroyed.
        mHolder = getHolder();
        mHolder.addCallback(this);
        mHolder.setType(SurfaceHolder.SURFACE_TYPE_PUSH_BUFFERS);
    }

    //SurfaceView callback
    public void surfaceCreated(SurfaceHolder holder) {
        if (mIsCameraOpen) return; // surfaceCreated can be called several times
        mIsCameraOpen = true;

        mFinished = false;
                
        // Find the ID of the camera
        int cameraId = 0;
        boolean frontCameraFound = false;
        Camera.CameraInfo cameraInfo = new Camera.CameraInfo();
        for (int i = 0; i < Camera.getNumberOfCameras(); i++) {
            Camera.getCameraInfo(i, cameraInfo);
            //if (cameraInfo.facing == Camera.CameraInfo.CAMERA_FACING_BACK)
            if (cameraInfo.facing == Camera.CameraInfo.CAMERA_FACING_FRONT) {
                cameraId = i;
                frontCameraFound = true;
            }
        }
        
        if (frontCameraFound) {
            mCamera = Camera.open(cameraId);
        } else {
            mCamera = Camera.open();
        }
        
        try {
            mCamera.setPreviewDisplay(holder);
            
            // Preview callback used whenever new viewfinder frame is available
            mCamera.setPreviewCallback(new PreviewCallback() {
                public void onPreviewFrame(byte[] data, Camera camera) {
                    if ( (mDraw == null) || mFinished )
                        return;
        
                    if (mDraw.mYUVData == null) {
                        // Initialize the draw-on-top companion
                        Camera.Parameters params = camera.getParameters();
                        mDraw.mImageWidth = params.getPreviewSize().width;
                        mDraw.mImageHeight = params.getPreviewSize().height;
                        mDraw.mRGBData = new byte[3 * mDraw.mImageWidth * mDraw.mImageHeight]; 
                        mDraw.mYUVData = new byte[data.length];            
                    }
    
                    // Pass YUV data to draw-on-top companion
                    System.arraycopy(data, 0, mDraw.mYUVData, 0, data.length);
                    mDraw.invalidate();
                }
            });
        } 
        catch (Exception exception) {
            AlertDialog.Builder builder = new AlertDialog.Builder(mContext);
            builder.setMessage("Cannot open camera" )
                .setPositiveButton("Ok", new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialogInterface, int i) {
                        android.os.Process.killProcess(android.os.Process.myPid());
                    }
                })
                .show();
            if (mCamera != null) {
                mCamera.release();
                mCamera = null;
            }
        }
    }


    public void releaseCallbacks() {
        if (mCamera != null) {
            mCamera.setPreviewCallback(null);
        }
        if (mHolder != null) {
            mHolder.removeCallback(this);
        }
        mDraw = null;
        mHolder = null;
    }

    //SurfaceView callback
    public void surfaceDestroyed(SurfaceHolder holder) {
        // Surface will be destroyed when we return, so stop the preview.
        // Because the CameraDevice object is not a shared resource, it's very
        // important to release it when the activity is paused.
        mFinished = true;
        if (mCamera != null) {
            mCamera.setPreviewCallback(null);
            mCamera.stopPreview();
            mCamera.release();
            mCamera = null;
        }

        mIsCameraOpen = false;
        mIsPreviewStarted = false;
    }
    
    //SurfaceView callback, configuring camera
    public void surfaceChanged(SurfaceHolder holder, int format, int w, int h) {
        if (mCamera == null) return;
        
        // Now that the size is known, set up the camera parameters and begin
        // the preview.
        Camera.Parameters parameters = mCamera.getParameters();

        //Keep uncommented to work correctly on phones:
        //This is an undocumented although widely known feature
        /**/
        if (this.getResources().getConfiguration().orientation != Configuration.ORIENTATION_LANDSCAPE) {
            parameters.set("orientation", "portrait");
            mCamera.setDisplayOrientation(90); // For Android 2.2 and above
            mDraw.rotated = true;
        } else {
            parameters.set("orientation", "landscape");
            mCamera.setDisplayOrientation(0); // For Android 2.2 and above
        }
        /**/
        
        // choose preview size closer to 640x480 for optimal performance
        List<Size> supportedSizes = parameters.getSupportedPreviewSizes();
        int width = 0;
        int height = 0;
        for (Size s: supportedSizes) {
            if ((width - 640)*(width - 640) + (height - 480)*(height - 480) > 
                    (s.width - 640)*(s.width - 640) + (s.height - 480)*(s.height - 480)) {
                width = s.width;
                height = s.height;
            }
        }
                
        //try to set preferred parameters
        try {
            if (width*height > 0) {
                parameters.setPreviewSize(width, height);
            }
            //parameters.setPreviewFrameRate(10);
            parameters.setSceneMode(Camera.Parameters.SCENE_MODE_PORTRAIT);
            parameters.setFocusMode(Camera.Parameters.FOCUS_MODE_CONTINUOUS_VIDEO);
            mCamera.setParameters(parameters);
        } catch (Exception ignored) {
        }

        if (!mIsPreviewStarted) {
            mCamera.startPreview();
            mIsPreviewStarted = true;
        }
        
        parameters = mCamera.getParameters();
        Camera.Size previewSize = parameters.getPreviewSize();
        makeResizeForCameraAspect(1.0f / ((1.0f * previewSize.width) / previewSize.height));
    }
    
    private void makeResizeForCameraAspect(float cameraAspectRatio){
        LayoutParams layoutParams = this.getLayoutParams();
        int matchParentWidth = this.getWidth();           
        int newHeight = (int)(matchParentWidth/cameraAspectRatio);
        if (newHeight != layoutParams.height) {
            layoutParams.height = newHeight;
            layoutParams.width = matchParentWidth;    
            this.setLayoutParams(layoutParams);
            this.invalidate();
        }        
    }
} // end of Preview class
